/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc.test;

import java.util.Iterator;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordReader;
import org.eclipse.smila.ipc.bon.BinaryToken;

/**
 * Test class for {@link IpcRecordReader} class.
 * 
 * @author aweber
 */
public class TestIpcRecordReader extends TestCase {

  private final IpcRecordReader _ipcRecordReader = new IpcRecordReader();

  /** BON -> empty Record. */
  public void testReadEmptyRecordFromBon() throws Exception {
    final byte[] bon =
      new byte[] { BinaryToken.OBJECT_START.byteValue(), BinaryToken.MAPPING_START.byteValue(),
        BinaryToken.MAPPING_END.byteValue(), BinaryToken.OBJECT_END.byteValue() };
    final Record expected = DataFactory.DEFAULT.createRecord();
    final Record parsed = _ipcRecordReader.readBinaryObject(bon);
    assertEquals(expected.getMetadata(), parsed.getMetadata());
    assertFalse(parsed.getAttachmentNames().hasNext());
  }

  /** BON -> Record with attachment. */
  public void testReadRecordWithAttachmentFromBon() throws Exception {
    final byte[] bon =
      new byte[] { BinaryToken.OBJECT_START.byteValue(), BinaryToken.MAPPING_START.byteValue(),
        BinaryToken.MAPPING_END.byteValue(), BinaryToken.ATTACHMENTS_START.byteValue(),
        BinaryToken.SCALAR_STRING_1.byteValue(), (byte) 1, (byte) 88, BinaryToken.BINARY_1.byteValue(), (byte) 1,
        (byte) 7, BinaryToken.ATTACHMENTS_END.byteValue(), BinaryToken.OBJECT_END.byteValue() };
    final Record expected = DataFactory.DEFAULT.createRecord();
    final Record parsed = _ipcRecordReader.readBinaryObject(bon);
    assertEquals(expected.getMetadata(), parsed.getMetadata());
    final Iterator<String> it = parsed.getAttachmentNames();
    assertTrue(it.hasNext());
    final String parsedAttachmentName = it.next();
    assertEquals("X", parsedAttachmentName); // X <=> (byte) 88
    assertEquals(1, parsed.getAttachmentAsBytes(parsedAttachmentName).length);
    assertFalse(it.hasNext());
  }

  /** JSON -> empty Record. */
  public void testReadEmptyRecordFromJson() throws Exception {
    final String json = "{}";
    final Record expected = DataFactory.DEFAULT.createRecord();
    final Record parsed = _ipcRecordReader.readJsonObject(json);
    assertEquals(expected.getMetadata(), parsed.getMetadata());
    assertFalse(parsed.getAttachmentNames().hasNext());
  }

  /** JSON -> Record with metadata. */
  public void testReadRecordWithMetadataFromJson() throws Exception {
    final String json = "{ \"test\":true, \"i\":5 }";
    final Record expected = DataFactory.DEFAULT.createRecord();
    expected.getMetadata().put("test", true);
    expected.getMetadata().put("i", 5);
    final Record parsed = _ipcRecordReader.readJsonObject(json);
    assertEquals(expected.getMetadata(), parsed.getMetadata());
    assertFalse(parsed.getAttachmentNames().hasNext());
  }

  /** JSON -> Record with attachments. */
  public void testReadRecordWithAttachmentsFromJson() throws Exception {
    final String json = "{ \"_attachmentNames\":[\"attachment-1\",\"attachment-2\"] }";
    final Record parsed = _ipcRecordReader.readJsonObject(json);
    assertTrue(parsed.hasAttachment("attachment-1"));
    assertTrue(parsed.hasAttachment("attachment-2"));
    assertFalse(parsed.hasAttachment("bla"));
  }

}
