/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc.test;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.ipc.bon.BinaryToken;

/**
 * Test class for {@link IpcRecordWriter} class.
 * 
 * @author aweber
 */
public class TestIpcRecordWriter extends TestCase {

  private IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /** empty Record -> JSON. */
  public void testWriteEmptyRecordToJSON() throws Exception {
    final Record testRecord = DataFactory.DEFAULT.createRecord();
    final String json = _ipcRecordWriter.writeJsonObject(testRecord);
    assertEquals("{}", json);
  }

  /** Record with metadata -> JSON. */
  public void testWriteRecordWithMetadataToJson() throws Exception {
    final Record testRecord = DataFactory.DEFAULT.createRecord();
    testRecord.getMetadata().put("test", true);
    testRecord.getMetadata().put("i", 5);
    final String json = _ipcRecordWriter.writeJsonObject(testRecord);
    assertEquals("{\"test\":true,\"i\":5}", json);
  }

  /** Record with attachments -> JSON. */
  public void testWriteRecordWithAttachmentsToJson() throws Exception {
    final Record testRecord = DataFactory.DEFAULT.createRecord();
    testRecord.setAttachment("attachment-1", new byte[0]);
    testRecord.setAttachment("attachment-2", new byte[] { 1, 2, 3 });
    final String json = _ipcRecordWriter.writeJsonObject(testRecord);
    assertEquals("{\"_attachmentNames\":[\"attachment-1\",\"attachment-2\"]}", json);
  }

  /** empty Record -> BON. */
  public void testWriteEmptyRecordToBon() throws Exception {
    final Record testRecord = DataFactory.DEFAULT.createRecord();
    final byte[] bon = _ipcRecordWriter.writeBinaryObject(testRecord);
    assertEquals(BinaryToken.OBJECT_START.byteValue(), bon[0]);
    assertEquals(BinaryToken.MAPPING_START.byteValue(), bon[1]);
    assertEquals(BinaryToken.MAPPING_END.byteValue(), bon[2]);
    assertEquals(BinaryToken.OBJECT_END.byteValue(), bon[3]);
  }

  /** Record with metadata -> BON. */
  public void testWriteRecordWithMetadataToBon() throws Exception {
    final Record testRecord = DataFactory.DEFAULT.createRecord();
    testRecord.getMetadata().put("test", true);
    testRecord.getMetadata().put("i", 5);
    final byte[] bon = _ipcRecordWriter.writeBinaryObject(testRecord);
    assertEquals(BinaryToken.OBJECT_START.byteValue(), bon[0]);
    assertEquals(BinaryToken.MAPPING_START.byteValue(), bon[1]);

    assertEquals(BinaryToken.SCALAR_STRING_1.byteValue(), bon[2]);
    assertEquals((byte) 4, bon[3]); // "test" string has length 4
    // byte[3]-[7] are the string value
    assertEquals(BinaryToken.SCALAR_BOOL_TRUE.byteValue(), bon[8]);

    assertEquals(BinaryToken.SCALAR_STRING_1.byteValue(), bon[9]);
    assertEquals((byte) 1, bon[10]); // "i" string has length 1
    // byte[11] is the string value
    assertEquals(BinaryToken.SCALAR_INT_1.byteValue(), bon[12]);
    assertEquals((byte) 5, bon[13]);

    assertEquals(BinaryToken.MAPPING_END.byteValue(), bon[14]);
    assertEquals(BinaryToken.OBJECT_END.byteValue(), bon[15]);
  }

  /** Record with attachments -> JSON. */
  public void testWriteRecordWithAttachmentToBon() throws Exception {
    final Record testRecord = DataFactory.DEFAULT.createRecord();
    testRecord.setAttachment("a", new byte[0]);
    final byte[] bon = _ipcRecordWriter.writeBinaryObject(testRecord);
    assertEquals(BinaryToken.OBJECT_START.byteValue(), bon[0]);
    assertEquals(BinaryToken.MAPPING_START.byteValue(), bon[1]);
    assertEquals(BinaryToken.MAPPING_END.byteValue(), bon[2]);
    assertEquals(BinaryToken.ATTACHMENTS_START.byteValue(), bon[3]);
    assertEquals(BinaryToken.SCALAR_STRING_1.byteValue(), bon[4]);
    assertEquals((byte) 1, bon[5]); // "a" string has length 1
    // byte[6] is the string value
    assertEquals(BinaryToken.BINARY_1.byteValue(), bon[7]);
    assertEquals((byte) 0, bon[8]); // attachment has length 0    
    assertEquals(BinaryToken.ATTACHMENTS_END.byteValue(), bon[9]);
    assertEquals(BinaryToken.OBJECT_END.byteValue(), bon[10]);
  }
}
