/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc.test;

import java.io.UnsupportedEncodingException;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ValueFormatHelper;
import org.eclipse.smila.datamodel.ipc.IpcRecordReader;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;

/** Test Record (de-)serialization. */
public class TestRecordSerialization extends TestCase {

  /** Constant for attribute Title. */
  public static final String ATT_TITLE = "Title";

  /** Constant for attribute Author. */
  public static final String ATT_AUTHOR = "Author";

  /** Constant for attribute Date. */
  public static final String ATT_DATE = "Date";

  /** Constant for attribute Date. */
  public static final String ATT_SIZE = "Size";

  /** Constant for attribute Similarity. */
  public static final String ATT_SIMILARITY = "Similarity";

  /** Constant for attribute Flag. */
  public static final String ATT_FLAG = "Flag";

  /** Constant for attribute singleObject. */
  public static final String ATT_SINGLEOBJECT = "singleObject";

  /** Constant for attribute multiObject. */
  public static final String ATT_MULTIOBJECT = "multiObject";

  /** Constant for attribute sequenceObject. */
  public static final String ATT_SEQUENCEOBJECT = "sequenceObject";

  private final IpcRecordWriter _writer = new IpcRecordWriter();

  private final IpcRecordReader _reader = new IpcRecordReader();

  /** test record <-> JSON (de-)serialization. */
  public void testJsonSerialization() throws Exception {
    final Record record = createRecord();
    //System.out.println(XmlSerializationUtils.serialize2string(record));

    // convert record to JSON
    final String s = _writer.writeJsonObject(record);
    assertNotNull(s);

    // re-convert JSON to record
    final Record parsedRecord = _reader.readJsonObject(s);
    assertNotNull(parsedRecord);
    //System.out.println(XmlSerializationUtils.serialize2string(parsedRecord));

    compareJsonConvertedRecords(record, parsedRecord);
  }

  /** test record <-> BON (de-)serialization. */
  public void testBON() throws Exception {
    final Record record = createRecord();
    //System.out.println(XmlSerializationUtils.serialize2string(record));

    // convert record to byte array
    final byte[] bytes = _writer.writeBinaryObject(record);
    assertNotNull(bytes);

    // re-convert byte array to record
    final Record parsedRecord = _reader.readBinaryObject(bytes);
    assertNotNull(parsedRecord);
    //System.out.println(XmlSerializationUtils.serialize2string(parsedRecord));

    // compare original and parsed metadata
    compareBonConvertedRecords(record, parsedRecord);
  }

  /** test that JSON pretty printing can be enabled in writer. */
  public void testJsonPrettyPrinting() throws Exception {
    final IpcRecordWriter prettyWriter = new IpcRecordWriter(true);
    final Record record = createRecord();
    final String prettyString = prettyWriter.writeJsonObject(record);
    assertTrue(prettyString.indexOf('\n') > 0);
    assertTrue(prettyString.indexOf("  ") > 0);
    final String oneLineString = _writer.writeJsonObject(record);
    assertTrue(oneLineString.indexOf('\n') < 0);
    assertTrue(oneLineString.indexOf("  ") < 0);
  }

  /** BON converted records have attachment/binary support. */
  private void compareBonConvertedRecords(final Record record, final Record parsedRecord) throws Exception {
    compareRecordMetadata(record, parsedRecord);
    final Iterator<String> iter = record.getAttachmentNames();
    while (iter.hasNext()) {
      final String attName = iter.next();
      assertTrue(parsedRecord.hasAttachment(attName));
      assertEquals(new String(record.getAttachmentAsBytes(attName)), new String(parsedRecord.getAttachmentAsBytes(attName)));
    }
  }

  /** helper method to compare the original and the (de-)serialized record. */
  private void compareJsonConvertedRecords(final Record record, final Record parsedRecord) throws Exception {
    compareRecordMetadata(record, parsedRecord);

    // attachments are not part of JSON, but their names are
    final Set<String> attachmentNamesExpected = new HashSet<String>();
    final Set<String> attachmentNamesParsed = new HashSet<String>();
    Iterator<String> iter = record.getAttachmentNames();
    while (iter.hasNext()) {
      attachmentNamesExpected.add(iter.next());
    }
    iter = parsedRecord.getAttachmentNames();
    while (iter.hasNext()) {
      final String attName = iter.next();
      attachmentNamesParsed.add(attName);
      assertNull(parsedRecord.getAttachmentAsBytes(attName));
    }
    assertEquals(attachmentNamesExpected, attachmentNamesParsed);
  }

  /** the metadata of the original record has to be equal to that of the parsed record. */
  private void compareRecordMetadata(final Record record, final Record parsedRecord) throws Exception {
    // compare original and id of parsed record
    assertEquals(record.getId(), parsedRecord.getId());

    // compare original and parsed metadata
    final AnyMap expected = record.getMetadata();
    final AnyMap metadata = parsedRecord.getMetadata();
    compareMap(expected, metadata);

    assertEquals(expected.get(ATT_SINGLEOBJECT).size(), metadata.get(ATT_SINGLEOBJECT).size());
    compareMap(expected.getMap(ATT_SINGLEOBJECT), metadata.getMap(ATT_SINGLEOBJECT));

    assertEquals(expected.get(ATT_MULTIOBJECT).size(), metadata.get(ATT_MULTIOBJECT).size());
    compareMap(expected.getSeq(ATT_MULTIOBJECT).getMap(0), metadata.getSeq(ATT_MULTIOBJECT).getMap(0));
    compareMap(expected.getSeq(ATT_MULTIOBJECT).getMap(1), metadata.getSeq(ATT_MULTIOBJECT).getMap(1));

    assertEquals(expected.get(ATT_SEQUENCEOBJECT).size(), metadata.get(ATT_SEQUENCEOBJECT).size());
    assertEquals(expected.getSeq(ATT_SEQUENCEOBJECT).get(0), metadata.getSeq(ATT_SEQUENCEOBJECT).get(0));
    assertEquals(expected.getSeq(ATT_SEQUENCEOBJECT).get(1), metadata.getSeq(ATT_SEQUENCEOBJECT).get(1));
  }

  /** compare metadata map objects. */
  private void compareMap(final AnyMap expected, final AnyMap parsed) {
    assertNotNull(parsed);
    assertEquals(expected.size(), parsed.size());
    assertEquals(expected.get(ATT_TITLE), parsed.get(ATT_TITLE));

    assertTrue(parsed.get(ATT_DATE).isDate()); // JSON doesn't support date type but our serialization should've
                                               // converted it.
    assertTrue(parsed.getDateValue(ATT_DATE) instanceof Date);
    assertEquals(ValueFormatHelper.INSTANCE.formatDate(expected.getDateValue(ATT_DATE)),
      ValueFormatHelper.INSTANCE.formatDate(parsed.getDateValue(ATT_DATE)));
    assertEquals(expected.get(ATT_SIZE), parsed.get(ATT_SIZE));
    assertEquals(expected.get(ATT_SIMILARITY), parsed.get(ATT_SIMILARITY));
    assertEquals(expected.get(ATT_FLAG), parsed.get(ATT_FLAG));
    assertEquals(expected.get(ATT_AUTHOR), parsed.get(ATT_AUTHOR));
  }

  /** Record used for testing. */
  private Record createRecord(final String... attachmentNames) throws UnsupportedEncodingException {
    final String idValue = "0815";
    final String source = "testSource";
    final String text = "Some test text without meaning.";

    final Record record = DataFactory.DEFAULT.createRecord(idValue, source);
    final AnyMap metadata = createMap();

    // Map value
    metadata.put(ATT_SINGLEOBJECT, createMap());

    // Seq of Map
    metadata.add(ATT_MULTIOBJECT, createMap());
    metadata.add(ATT_MULTIOBJECT, createMap());

    // Seq of Seq
    final AnySeq sequenceObjects = DataFactory.DEFAULT.createAnySeq();
    sequenceObjects.add(createSeq());
    sequenceObjects.add(createSeq());
    metadata.put(ATT_SEQUENCEOBJECT, sequenceObjects);

    record.getMetadata().putAll(metadata);

    for (final String attachment : attachmentNames) {
      record.setAttachment(attachment, text.getBytes());
    }
    return record;
  }

  /** AnySeq used for testing. */
  private AnySeq createSeq() {
    final AnySeq seq = DataFactory.DEFAULT.createAnySeq();
    seq.add(new Long(1));
    seq.add(new Long(0));
    seq.add(new Long(2));
    seq.add("Pinguin");
    return seq;
  }

  /** AnyMap used for testing. */
  private AnyMap createMap() {
    final AnyMap metadata = DataFactory.DEFAULT.createAnyMap();
    final String title = "test title";
    final String[] authors = { "karl", "heinz" };
    final Date date = new Date();
    final String text = "Some test text without meaning.";
    final int size = text.length();
    final double similarity = 0.75;
    final boolean flag = true;
    metadata.put(ATT_TITLE, title);
    for (final String author : authors) {
      metadata.add(ATT_AUTHOR, DataFactory.DEFAULT.createStringValue(author));
    }
    metadata.put(ATT_DATE, DataFactory.DEFAULT.createDateValue(date));
    metadata.put(ATT_SIZE, size);
    metadata.put(ATT_SIMILARITY, similarity);
    metadata.put(ATT_FLAG, flag);
    return metadata;
  }
}
