/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.datamodel.test;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;

/**
 * Test cases for the default DataFactory.
 */
public class TestDefaultDataFactory extends TestCase {

  /**
   * The DataFactory under test.
   */
  private final DataFactory _dataFactory = DataFactory.DEFAULT;

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createRecord()}.
   */
  public void testCreateRecord() {
    final Record record = _dataFactory.createRecord();
    assertNotNull(record);
    assertNull(record.getId());
    assertNull(record.getSource());
    assertNotNull(record.getMetadata());
    assertTrue(record.getMetadata().isEmpty());
    assertTrue(record.getMetadata().isMap());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createRecord(java.lang.String)}.
   */
  public void testCreateRecordString() {
    final Record record = _dataFactory.createRecord("id");
    assertNotNull(record);
    assertEquals("id", record.getId());
    assertNull(record.getSource());
    assertNotNull(record.getMetadata());
    assertNotNull(record.getMetadata().get("_recordid"));
    assertTrue(record.getMetadata().isMap());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createRecord(java.lang.String,
   * java.lang.String)}.
   */
  public void testCreateRecordStringString() {
    final Record record = _dataFactory.createRecord("id", "source");
    assertNotNull(record);
    assertEquals("id", record.getId());
    assertEquals("source", record.getSource());
    assertNotNull(record.getMetadata());
    assertNotNull(record.getMetadata().get("_recordid"));
    assertNotNull(record.getMetadata().get("_source"));
    assertTrue(record.getMetadata().isMap());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createAnyMap()}.
   */
  public void testCreateAnyMap() {
    final AnyMap anySeq = _dataFactory.createAnyMap();
    assertEquals(0, anySeq.size());
    assertEquals(ValueType.MAP, anySeq.getValueType());
    assertTrue(anySeq.isMap());
    assertFalse(anySeq.isSeq());
    assertFalse(anySeq.isString());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createAnySeq()}.
   */
  public void testCreateAnySeq() {
    final AnySeq anySeq = _dataFactory.createAnySeq();
    assertEquals(0, anySeq.size());
    assertEquals(ValueType.SEQ, anySeq.getValueType());
    assertTrue(anySeq.isSeq());
    assertFalse(anySeq.isMap());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createStringValue(java.lang.String)} .
   */
  public void testCreateStringValue() {
    final String inValue = "in string";
    final Value value = _dataFactory.createStringValue(inValue);
    assertEquals(ValueType.STRING, value.getValueType());
    assertEquals(inValue, value.asString());
    assertTrue(value.isString());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createBooleanValue(java.lang.Boolean)}.
   */
  public void testCreateBooleanValue() {
    final Boolean inValue = Boolean.TRUE;
    final Value value = _dataFactory.createBooleanValue(inValue);
    assertEquals(ValueType.BOOLEAN, value.getValueType());
    assertEquals(inValue, value.asBoolean());
    assertTrue(value.isBoolean());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createLongValue(java.lang.Long)}.
   */
  public void testCreateLongValueLong() {
    final Long inValue = 314L;
    final Value value = _dataFactory.createLongValue(inValue);
    assertEquals(ValueType.LONG, value.getValueType());
    assertEquals(inValue, value.asLong());
    assertTrue(value.isLong());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createLongValue(int)}.
   */
  public void testCreateLongValueInt() {
    final Integer inValue = 314;
    final Value value = _dataFactory.createLongValue(inValue.intValue());
    assertEquals(ValueType.LONG, value.getValueType());
    assertEquals(inValue.longValue(), value.asLong().longValue());
    assertTrue(value.isLong());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createDoubleValue(java.lang.Double)} .
   */
  public void testCreateDoubleValueDouble() {
    final Double inValue = 3.14d;
    final Value value = _dataFactory.createDoubleValue(inValue);
    assertEquals(ValueType.DOUBLE, value.getValueType());
    assertEquals(inValue, value.asDouble());
    assertTrue(value.isDouble());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createDoubleValue(float)}.
   */
  public void testCreateDoubleValueFloat() {
    final Float inValue = 3.14f;
    final Value value = _dataFactory.createDoubleValue(inValue.floatValue());
    assertEquals(ValueType.DOUBLE, value.getValueType());
    assertTrue(inValue - 0.01 < value.asDouble() && inValue + 0.01 > value.asDouble());
    assertTrue(value.isDouble());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createDateValue(java.util.Date)}.
   */
  public void testCreateDateValue() {
    final Date inValue = new Date();
    final Value value = _dataFactory.createDateValue(inValue);
    assertEquals(ValueType.DATE, value.getValueType());
    assertEquals(inValue, value.asDate());
    assertTrue(value.isDate());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#createDateTimeValue(java.util.Date)} .
   */
  public void testCreateDateTimeValue() {
    final Date inValue = new Date();
    final Value value = _dataFactory.createDateTimeValue(inValue);
    assertEquals(ValueType.DATETIME, value.getValueType());
    assertEquals(inValue, value.asDateTime());
    assertTrue(value.isDateTime());
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#parseFromString(java.lang.String,
   * java.lang.String)} .
   */
  public void testParseFromString() {
    Value value = _dataFactory.parseFromString("test", (String) null);
    assertEquals(ValueType.STRING, value.getValueType());
    value = _dataFactory.parseFromString("test", "string");
    assertEquals(ValueType.STRING, value.getValueType());
    value = _dataFactory.parseFromString("test", "String");
    assertEquals(ValueType.STRING, value.getValueType());
    value = _dataFactory.parseFromString("test", "STRING");
    assertEquals(ValueType.STRING, value.getValueType());
    value = _dataFactory.parseFromString("1", "long");
    assertEquals(ValueType.LONG, value.getValueType());
    assertEquals(1, value.asLong().longValue());
    value = _dataFactory.parseFromString("-3", "long");
    assertEquals(ValueType.LONG, value.getValueType());
    assertEquals(-3, value.asLong().longValue());
    value = _dataFactory.parseFromString(String.valueOf(Long.MAX_VALUE), "LONG");
    assertEquals(ValueType.LONG, value.getValueType());
    assertEquals(Long.MAX_VALUE, value.asLong().longValue());
    value = _dataFactory.parseFromString("3.1415926535897932384626433832795", "double");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    double parsedValue = value.asDouble().doubleValue();
    assertTrue(3.14 < parsedValue && parsedValue < 3.15);
    value = _dataFactory.parseFromString("5", "double");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    parsedValue = value.asDouble().doubleValue();
    assertEquals(5.0d, parsedValue);
    value = _dataFactory.parseFromString("3.141f", "double");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    parsedValue = value.asDouble().doubleValue();
    assertTrue(3.14 < parsedValue && parsedValue < 3.15);
    value = _dataFactory.parseFromString("-3.141d", "double");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    parsedValue = value.asDouble().doubleValue();
    assertTrue(-3.14 > parsedValue && parsedValue > -3.15);
    value = _dataFactory.parseFromString("3141e-3", "Double");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    parsedValue = value.asDouble().doubleValue();
    assertTrue(3.14 < parsedValue && parsedValue < 3.15);
    value = _dataFactory.parseFromString("NaN", "DOUBLE");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    assertEquals(Double.NaN, value.asDouble());
    value = _dataFactory.parseFromString("Infinity", "DOUBLE");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    assertEquals(Double.POSITIVE_INFINITY, value.asDouble());
    value = _dataFactory.parseFromString("-Infinity", "DOUBLE");
    assertEquals(ValueType.DOUBLE, value.getValueType());
    assertEquals(Double.NEGATIVE_INFINITY, value.asDouble());
    value = _dataFactory.parseFromString("True", "boolean");
    assertEquals(ValueType.BOOLEAN, value.getValueType());
    assertEquals(Boolean.TRUE, value.asBoolean());
    value = _dataFactory.parseFromString("FaLsE", "BOOLEAN");
    assertEquals(ValueType.BOOLEAN, value.getValueType());
    assertEquals(Boolean.FALSE, value.asBoolean());
    value = _dataFactory.parseFromString("2011-03-09T11:31:00+0100", "dateTime");
    assertEquals(ValueType.DATETIME, value.getValueType());
    Calendar cal = new GregorianCalendar();
    cal.setTime(value.asDateTime());
    assertEquals(31, cal.get(Calendar.MINUTE));
    assertEquals(9, cal.get(Calendar.DAY_OF_MONTH));
    assertEquals(2011, cal.get(Calendar.YEAR));
    try {
      value = _dataFactory.parseFromString("2011-03-09T11:31:00+0100", "date");
      fail("exception expected");
    } catch (final InvalidValueTypeException ex) {
      ; // OK
    }
    value = _dataFactory.parseFromString("2011-03-09", "Date");
    assertEquals(ValueType.DATE, value.getValueType());
    cal = new GregorianCalendar();
    cal.setTime(value.asDate());
    assertEquals(0, cal.get(Calendar.MINUTE));
    assertEquals(9, cal.get(Calendar.DAY_OF_MONTH));
    assertEquals(2, cal.get(Calendar.MONTH));
    assertEquals(2011, cal.get(Calendar.YEAR));

    try {
      _dataFactory.parseFromString("x", "long");
      fail("exception expected");
    } catch (InvalidValueTypeException ex) {
      ex = null;
    } catch (final Exception ex) {
      fail("wrong exception caught");
    }
    try {
      _dataFactory.parseFromString("1", "int");
      fail("exception expected");
    } catch (InvalidValueTypeException ex) {
      ex = null;
    } catch (final Exception ex) {
      fail("wrong exception caught");
    }
  }

  /**
   * Test method for { org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#parseFromObject(java.lang.Object)}.
   */
  public void testParseFromObject() {
    Value value = _dataFactory.autoConvertValue("test");
    assertEquals(ValueType.STRING, value.getValueType());
    value = _dataFactory.autoConvertValue(1L);
    assertEquals(ValueType.LONG, value.getValueType());
    value = _dataFactory.autoConvertValue(Short.valueOf((short) 1));
    assertEquals(ValueType.LONG, value.getValueType());
    value = _dataFactory.autoConvertValue(Math.PI);
    assertEquals(ValueType.DOUBLE, value.getValueType());
    value = _dataFactory.autoConvertValue((float) Math.PI);
    assertEquals(ValueType.DOUBLE, value.getValueType());
    value = _dataFactory.autoConvertValue(new BigDecimal(Math.PI));
    assertEquals(ValueType.DOUBLE, value.getValueType());
    value = _dataFactory.autoConvertValue(true);
    assertEquals(ValueType.BOOLEAN, value.getValueType());
    value = _dataFactory.autoConvertValue(new Date());
    assertEquals(ValueType.DATETIME, value.getValueType());
    try {
      _dataFactory.autoConvertValue(new byte[0]);
      fail("exception expected");
    } catch (InvalidValueTypeException ex) {
      ex = null;
    } catch (final Exception ex) {
      fail("wrong exception caught");
    }
  }

  /** some simple tests for autodetection of dates/datetimes and correct reproduction of the original string. */
  public void testTryDateTimeParsing() {
    final Value dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01");
    assertEquals(ValueType.DATE, dateValue.getValueType());
    assertEquals("2010-01-01", dateValue.asString());
    Value timestampValueNoMillis = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T12:34:56+0100");
    assertEquals(ValueType.DATETIME, timestampValueNoMillis.getValueType());
    assertEquals("2010-01-01T12:34:56+0100", timestampValueNoMillis.asString());
    timestampValueNoMillis = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T12:34:56-0100");
    assertEquals(ValueType.DATETIME, timestampValueNoMillis.getValueType());
    assertEquals("2010-01-01T12:34:56-0100", timestampValueNoMillis.asString());
    timestampValueNoMillis = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T12:34:56-0000");
    assertEquals(ValueType.DATETIME, timestampValueNoMillis.getValueType());
    assertEquals("2010-01-01T12:34:56-0000", timestampValueNoMillis.asString());
    final Value timestampValueWithMillis =
      _dataFactory.tryDateTimestampParsingFromString("2010-01-01T12:34:56.000+0100");
    assertEquals(ValueType.DATETIME, timestampValueWithMillis.getValueType());
    assertEquals("2010-01-01T12:34:56.000+0100", timestampValueWithMillis.asString());
    final Value timestampValueZulu = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T12:34:56.000Z");
    assertEquals(ValueType.DATETIME, timestampValueZulu.getValueType());
    assertEquals("2010-01-01T12:34:56.000Z", timestampValueZulu.asString());
    final Value timestampValueNoMillisZulu = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T12:34:56Z");
    assertEquals(ValueType.DATETIME, timestampValueNoMillisZulu.getValueType());
    assertEquals("2010-01-01T12:34:56Z", timestampValueNoMillisZulu.asString());
    final Value justAString = _dataFactory.tryDateTimestampParsingFromString("just a string");
    assertEquals(ValueType.STRING, justAString.getValueType());
    assertEquals("just a string", justAString.asString());
  }

  /** some simple tests for wrong dates. */
  public void testTryWrongDates() {
    // wrong year
    Value dateValue = _dataFactory.tryDateTimestampParsingFromString("20101-01-01");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20101-01-01", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("201-01-01");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("201-01-01", dateValue.asString());
    // wrong month
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-13-01");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-13-01", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-001-01");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-001-01", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-1-01");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-1-01", dateValue.asString());
    // wrong day
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-11-31");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-11-31", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-40");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-40", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-001");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-001", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-1");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-1", dateValue.asString());
    // wrong day and month with correct length of pattern
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-001-1");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-001-1", dateValue.asString());
    // wrong year and day with correct length of pattern
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20100-01-1");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20100-01-1", dateValue.asString());
    // wrong year and month with correct length of pattern
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20100-1-10");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20100-1-10", dateValue.asString());
  }

  /** some simple tests for wrong date times. */
  public void testTryWrongDateTimes() {
    // wrong year
    Value dateValue = _dataFactory.tryDateTimestampParsingFromString("20101-01-01T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20101-01-01T12:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("201-01-01T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("201-01-01T12:34:56Z", dateValue.asString());
    // wrong month
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-13-01T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-13-01T12:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-1-01T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-1-01T12:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-001-01T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-001-01T12:34:56Z", dateValue.asString());
    // wrong day
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-02-30T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-02-30T12:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-02-1T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-02-1T12:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-02-001T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-02-001T12:34:56Z", dateValue.asString());
    // wrong day and month with correct length of pattern
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-1-001T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-1-001T12:34:56Z", dateValue.asString());
    // wrong year and month with correct length of pattern
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20100-1-01T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20100-1-01T12:34:56Z", dateValue.asString());
    // wrong year and day with correct length of pattern
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20100-01-1T12:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20100-01-1T12:34:56Z", dateValue.asString());
    // wrong hour
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T1:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T1:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T25:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T25:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T001:34:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    // wrong minutes
    assertEquals("2010-01-01T001:34:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:4:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:4:56Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:64:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:64:56Z", dateValue.asString());
    // wrong hours and minutes with correct pattern length
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T1:004:56Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T1:004:56Z", dateValue.asString());
    // wrong seconds
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:04:001Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:04:001Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:6Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:6Z", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:61Z");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:61Z", dateValue.asString());
    // wrong year
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20101-01-01T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20101-01-01T12:34:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("201-01-01T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("201-01-01T12:34:56+0100", dateValue.asString());
    // wrong month
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-13-01T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-13-01T12:34:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-1-01T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-1-01T12:34:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-001-01T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-001-01T12:34:56+0100", dateValue.asString());
    // wrong day
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-02-30T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-02-30T12:34:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-02-1T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-02-1T12:34:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-02-001T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-02-001T12:34:56+0100", dateValue.asString());
    // wrong month and day with correct pattern length
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-1-001T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-1-001T12:34:56+0100", dateValue.asString());
    // wrong year and month with correct pattern length
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20100-1-01T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20100-1-01T12:34:56+0100", dateValue.asString());
    // wrong year and day with correct pattern length
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20100-01-1T12:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20100-01-1T12:34:56+0100", dateValue.asString());
    // wrong year and hour with correct pattern length
    dateValue = _dataFactory.tryDateTimestampParsingFromString("20100-01-01T2:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("20100-01-01T2:34:56+0100", dateValue.asString());
    // wrong hours
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T1:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T1:34:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T25:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T25:34:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T001:34:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T001:34:56+0100", dateValue.asString());
    // wrong minutes
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:4:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:4:56+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:64:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:64:56+0100", dateValue.asString());
    // wrong seconds
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:04:001+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:04:001+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:6+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:6+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:6+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:6+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:61+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:61+0100", dateValue.asString());
    // wrong time zone
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01+100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01+100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01-1400");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01-1400", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01+1500");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01+1500", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01+00100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01+00100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01.00+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01.00+0100", dateValue.asString());
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01.1111+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01.1111+0100", dateValue.asString());
    // wrong hours and minutes with correct pattern length
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T1:004:56+0100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T1:004:56+0100", dateValue.asString());
    // wrong month and time zone with correct pattern length
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-1-01T1:004:56+00100");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-1-01T1:004:56+00100", dateValue.asString());
    // no time zone
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01.111");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01.111", dateValue.asString());
    // wrong time zone
    dateValue = _dataFactory.tryDateTimestampParsingFromString("2010-01-01T01:44:01.111*0900");
    assertEquals(ValueType.STRING, dateValue.getValueType());
    assertEquals("2010-01-01T01:44:01.111*0900", dateValue.asString());
  }

  /**
   * Test method for {
   * org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#cloneAny(org.eclipse.smila.datamodel.Any)}.
   */
  public void testCloneAny() {
    Value source = _dataFactory.createBooleanValue(Boolean.TRUE);
    assertEquals(source, _dataFactory.cloneAny(source));
    source = _dataFactory.createStringValue("in string");
    assertEquals(source, _dataFactory.cloneAny(source));
    source = _dataFactory.createDateTimeValue(new Date());
    assertEquals(source, _dataFactory.cloneAny(source));
    source = _dataFactory.createDateValue(new Date());
    assertEquals(source, _dataFactory.cloneAny(source));
    source = _dataFactory.createDoubleValue(1.5d);
    assertEquals(source, _dataFactory.cloneAny(source));
    source = _dataFactory.createLongValue(Long.MIN_VALUE);
    assertEquals(source, _dataFactory.cloneAny(source));
    final AnyMap sourceMap = _dataFactory.createAnyMap();
    sourceMap.put("blabla", source);
    assertEquals(sourceMap, _dataFactory.cloneAny(sourceMap));
    final AnySeq anySeq = _dataFactory.createAnySeq();
    anySeq.add(source);
    anySeq.add(sourceMap);
    assertEquals(anySeq, _dataFactory.cloneAny(anySeq));
  }

  /**
   * Test method for {
   * org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#cloneAnyMap(org.eclipse.smila.datamodel.AnyMap)}.
   */
  public void testCloneAnyMap() {
    final Any val1 = _dataFactory.createLongValue(Long.MIN_VALUE);
    final Any val2 = _dataFactory.createStringValue("string");
    final AnyMap val3 = _dataFactory.createAnyMap();
    final AnySeq val4 = _dataFactory.createAnySeq();
    val4.add(val1);
    val4.add(val2);
    val3.put("seq", val4);
    val3.put("String", val2);
    assertEquals(val3, _dataFactory.cloneAnyMap(val3));
  }

  /**
   * Test method for {
   * org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl#cloneAnySeq(org.eclipse.smila.datamodel.AnySeq)}.
   */
  public void testCloneAnySeq() {
    final Any val1 = _dataFactory.createLongValue(Long.MIN_VALUE);
    final Any val2 = _dataFactory.createStringValue("string");
    final AnyMap val3 = _dataFactory.createAnyMap();
    final AnySeq val4 = _dataFactory.createAnySeq();
    val4.add(val1);
    val4.add(val2);
    val3.put("Long", val1);
    val3.put("String", val2);
    val4.add(val3);
    val4.add(val3);
    val4.add(val2);
    val4.add(val3);
    assertEquals(val4, _dataFactory.cloneAnySeq(val4));
  }

  /** test for cloneRecord(..., false) method. */
  public void testCloneRecordWithoutAttachments() {
    final Record record = _dataFactory.createRecord("id");
    record.getMetadata().put("string", "value");
    record.getMetadata().put("long", 42);
    record.getMetadata().put("double", Math.PI);
    record.getMetadata().put("boolean", true);
    record.setAttachment("attachment", "attachment".getBytes());
    final Record clone = _dataFactory.cloneRecord(record, "clone", false);
    assertNotSame(record, clone);
    assertEquals("clone", clone.getId());
    assertNotSame(record.getMetadata(), clone.getMetadata());
    assertNotSame(record.getMetadata().get("string"), clone.getMetadata().get("string"));
    assertNotSame(record.getMetadata().get("long"), clone.getMetadata().get("long"));
    assertNotSame(record.getMetadata().get("double"), clone.getMetadata().get("double"));
    assertNotSame(record.getMetadata().get("boolean"), clone.getMetadata().get("boolean"));
    assertEquals(record.getMetadata().get("string"), clone.getMetadata().get("string"));
    assertEquals(record.getMetadata().get("long"), clone.getMetadata().get("long"));
    assertEquals(record.getMetadata().get("double"), clone.getMetadata().get("double"));
    assertEquals(record.getMetadata().get("boolean"), clone.getMetadata().get("boolean"));
    assertFalse(clone.hasAttachments());
  }

  /** test for cloneRecord(..., true) method. */
  public void testCloneRecordWithAttachments() {
    final Record record = _dataFactory.createRecord("id");
    record.getMetadata().put("string", "value");
    record.getMetadata().put("long", 42);
    record.getMetadata().put("double", Math.PI);
    record.getMetadata().put("boolean", true);
    record.setAttachment("attachment", "attachment".getBytes());
    final Record clone = _dataFactory.cloneRecord(record, "clone", true);
    assertNotSame(record, clone);
    assertEquals("clone", clone.getId());
    assertNotSame(record.getMetadata(), clone.getMetadata());
    assertNotSame(record.getMetadata().get("string"), clone.getMetadata().get("string"));
    assertNotSame(record.getMetadata().get("long"), clone.getMetadata().get("long"));
    assertNotSame(record.getMetadata().get("double"), clone.getMetadata().get("double"));
    assertNotSame(record.getMetadata().get("boolean"), clone.getMetadata().get("boolean"));
    assertEquals(record.getMetadata().get("string"), clone.getMetadata().get("string"));
    assertEquals(record.getMetadata().get("long"), clone.getMetadata().get("long"));
    assertEquals(record.getMetadata().get("double"), clone.getMetadata().get("double"));
    assertEquals(record.getMetadata().get("boolean"), clone.getMetadata().get("boolean"));
    assertTrue(clone.hasAttachments());
    assertSame(record.getAttachmentAsBytes("attachment"), clone.getAttachmentAsBytes("attachment"));
  }

  /** test for createRecord(Record, false) method. */
  public void testCreateRecordFromRecordWithoutAttachments() {
    final Record record = _dataFactory.createRecord("id");
    record.getMetadata().put("string", "value");
    record.getMetadata().put("long", 42);
    record.getMetadata().put("double", Math.PI);
    record.getMetadata().put("boolean", true);
    record.setAttachment("attachment", "attachment".getBytes());
    final Record clone = _dataFactory.createRecord(record, false);
    assertNotSame(record, clone);
    assertNotSame(record.getMetadata(), clone.getMetadata());
    assertSame(record.getMetadata().get("_recordid"), clone.getMetadata().get("_recordid"));
    assertSame(record.getMetadata().get("string"), clone.getMetadata().get("string"));
    assertSame(record.getMetadata().get("long"), clone.getMetadata().get("long"));
    assertSame(record.getMetadata().get("double"), clone.getMetadata().get("double"));
    assertSame(record.getMetadata().get("boolean"), clone.getMetadata().get("boolean"));
    assertFalse(clone.hasAttachments());
  }

  /** test for createRecord(Record, true) method. */
  public void testCreateRecordFromRecordWithAttachments() {
    final Record record = _dataFactory.createRecord("id");
    record.getMetadata().put("string", "value");
    record.getMetadata().put("long", 42);
    record.getMetadata().put("double", Math.PI);
    record.getMetadata().put("boolean", true);
    record.setAttachment("attachment", "attachment".getBytes());
    final Record clone = _dataFactory.createRecord(record, true);
    assertNotSame(record, clone);
    assertNotSame(record.getMetadata(), clone.getMetadata());
    assertSame(record.getMetadata().get("_recordid"), clone.getMetadata().get("_recordid"));
    assertSame(record.getMetadata().get("string"), clone.getMetadata().get("string"));
    assertSame(record.getMetadata().get("long"), clone.getMetadata().get("long"));
    assertSame(record.getMetadata().get("double"), clone.getMetadata().get("double"));
    assertSame(record.getMetadata().get("boolean"), clone.getMetadata().get("boolean"));
    assertTrue(clone.hasAttachments());
    assertSame(record.getAttachmentAsBytes("attachment"), clone.getAttachmentAsBytes("attachment"));
  }

}
