/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.ipc.IpcFactory;
import org.eclipse.smila.ipc.IpcStreamReader;
import org.eclipse.smila.ipc.IpcToken;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.json.JsonFactory;

/**
 * Utility class for reading Any objects from binary (BON) and JSON objects.
 * 
 * @author aweber
 */
public class IpcAnyReader {

  /** Encoding to use for String (de)serialization. */
  public static final String ENCODING = "UTF-8";

  /** BON IPC factory. */
  private final IpcFactory _binaryFactory = new BinaryFactory();

  /** JSON IPC factory. */
  private final IpcFactory _jsonFactory = new JsonFactory();

  /**
   * @param binary
   * @return Any converted from BON object.
   * @throws IOException
   */
  public Any readBinaryObject(final byte[] binary) throws IOException {
    ByteArrayInputStream bais = null;
    try {
      bais = new ByteArrayInputStream(binary);
      return readBinaryStream(bais);
    } finally {
      IOUtils.closeQuietly(bais);
    }
  }

  /**
   * @param stream
   * @return Any converted from input stream
   * @throws IOException
   */
  public Any readBinaryStream(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _binaryFactory.newStreamReader(stream);
    return readStream(reader);
  }

  /**
   * @param json
   * @return Any converted from input JSON string.
   * @throws IOException
   */
  public Any readJsonObject(final String json) throws IOException {
    ByteArrayInputStream jsonStream = null;
    try {
      jsonStream = new ByteArrayInputStream(json.getBytes(ENCODING));
      return readJsonStream(jsonStream);
    } finally {
      IOUtils.closeQuietly(jsonStream);
    }
  }

  /**
   * @param stream
   * @return Any converted from input JSON stream.
   * @throws IOException
   */
  public Any readJsonStream(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _jsonFactory.newStreamReader(stream);
    try {
      return readStream(reader);
    } finally {
      reader.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * @param reader
   * @return Any converted from input reader's underlying stream.
   * @throws IOException
   */
  public Any readStream(final IpcStreamReader reader) throws IOException {
    IpcToken token = reader.nextToken();
    if (token == null) {
      return null;
    }
    if (token != IpcToken.OBJECT_START) {
      throw new IllegalStateException("Expected OBJECT_START Token; Token = " + token);
    }
    token = reader.nextToken();
    final Any any;
    if (token != IpcToken.OBJECT_END) {
      any = readAny(reader, token);
      token = reader.nextToken();
    } else {
      any = null;
    }
    if (token != IpcToken.OBJECT_END) {
      throw new IllegalStateException("Expected OBJECT_END Token; Token = " + token);
    }
    return any;
  }

  /**
   * helper method to read an Any.
   */
  private Any readAny(final IpcStreamReader reader, final IpcToken token) throws IOException {
    Any any = null;
    switch (token) {
      case MAPPING_START:
        any = readMap(reader);
        break;
      case SEQUENCE_START:
        any = readSeq(reader);
        break;
      default:
        any = readValue(reader, token);
        break;
    }
    return any;
  }

  /**
   * helper method to read an AnyMap.
   */
  AnyMap readMap(final IpcStreamReader reader) throws IOException {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    IpcToken token;
    while ((token = reader.nextToken()) != IpcToken.MAPPING_END) {
      if (token != IpcToken.SCALAR_STRING) { // expect mapping key
        throw new IllegalStateException("Expected SCALAR_STRING Token; Token = " + token);
      }
      final String key = reader.currentStringValue();
      map.put(key, readAny(reader, reader.nextToken()));
    }
    return map;
  }

  /**
   * helper method to read an AnySeq.
   */
  private AnySeq readSeq(final IpcStreamReader reader) throws IOException {
    final AnySeq anySeq = DataFactory.DEFAULT.createAnySeq();
    IpcToken token = null;
    while ((token = reader.nextToken()) != IpcToken.SEQUENCE_END) {
      anySeq.add(readAny(reader, token));
    }
    return anySeq;
  }

  /**
   * helper method to read a Value.
   */
  private Value readValue(final IpcStreamReader reader, final IpcToken token) throws IOException {
    Value value = null;
    switch (token) {
      case SCALAR_BOOL:
        value = DataFactory.DEFAULT.createBooleanValue(reader.currentBoolValue());
        break;
      case SCALAR_DOUBLE:
        value = DataFactory.DEFAULT.createDoubleValue(reader.currentDoubleValue());
        break;
      case SCALAR_INT:
        value = DataFactory.DEFAULT.createLongValue(reader.currentLongValue());
        break;
      case SCALAR_STRING:
      default:
        // can also be DATE and DATETIME, let the data factory guess
        value = DataFactory.DEFAULT.tryDateTimestampParsingFromString(reader.currentStringValue());
    }
    return value;
  }

}
