/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Map.Entry;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.ipc.IpcFactory;
import org.eclipse.smila.ipc.IpcStreamWriter;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.json.JsonFactory;

/**
 * Utility class for writing Any objects to binary (BON) and JSON.
 * 
 * @author aweber
 */
public class IpcAnyWriter {

  /** Encoding to use for String (de)serialization. */
  public static final String ENCODING = "UTF-8";

  /** BON IPC factory. */
  private final IpcFactory _binaryFactory = new BinaryFactory();

  /** JSON IPC factory. */
  private final IpcFactory _jsonFactory;

  /** */
  public IpcAnyWriter() {
    this(false);
  }

  /**
   * @param printPretty
   *          whether JSON output should be formatted.
   */
  public IpcAnyWriter(final boolean printPretty) {
    _jsonFactory = new JsonFactory(printPretty);
  }

  /**
   * @param any
   *          input Any object
   * @return input Any converted to BON
   * @throws IOException
   */
  public byte[] writeBinaryObject(final Any any) throws IOException {
    final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    writeBinaryStream(any, outputStream);
    return outputStream.toByteArray();
  }

  /**
   * @param any
   *          input
   * @param stream
   *          stream to write the output in BON format.
   * @throws IOException
   */
  public void writeBinaryStream(final Any any, final OutputStream stream) throws IOException {
    final IpcStreamWriter writer = _binaryFactory.newStreamWriter(stream);
    writeStream(any, writer);
  }

  /**
   * @param any
   *          input Any object
   * @return input Any converted to JSON
   * @throws IOException
   */
  public String writeJsonObject(final Any any) throws IOException {
    final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    writeJsonStream(any, outputStream);
    return new String(outputStream.toByteArray(), ENCODING);
  }

  /**
   * @param any
   *          input
   * @param stream
   *          stream to write the output as JSON.
   * @throws IOException
   */
  public void writeJsonStream(final Any any, final OutputStream stream) throws IOException {
    final IpcStreamWriter writer = _jsonFactory.newStreamWriter(stream);
    writeStream(any, writer);
  }

  /**
   * @param any
   *          input
   * @param writer
   *          writer with an underlying stream to write the ouput to.
   * @throws IOException
   */
  public void writeStream(final Any any, final IpcStreamWriter writer) throws IOException {
    try {
      writer.writeObjectStart();
      writeAny(writer, any);
      writer.writeObjectEnd();
    } finally {
      writer.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * helper method to write an Any object.
   */
  void writeAny(final IpcStreamWriter writer, final Any any) throws IOException {
    if (any != null) {
      if (any.isMap()) {
        writeMap(writer, (AnyMap) any);
      } else if (any.isSeq()) {
        writeSeq(writer, (AnySeq) any);
      } else if (any.isValue()) {
        writeValue(writer, (Value) any);
      }
    }
  }

  /**
   * helper method to write an AnyMap.
   */
  private void writeMap(final IpcStreamWriter writer, final AnyMap object) throws IOException {
    if (object != null) {
      writer.writeMappingStart();
      if (!object.isEmpty()) {
        for (final Entry<String, Any> childEntry : object.entrySet()) {
          final String childKey = childEntry.getKey();
          writer.writeMappingKey(childKey);
          final Any any = childEntry.getValue();
          writeAny(writer, any);
        }
      }
      writer.writeMappingEnd();
    }
  }

  /**
   * helper method to write an AnySeq.
   */
  private void writeSeq(final IpcStreamWriter writer, final AnySeq seq) throws IOException {
    writer.writeSequenceStart();
    final Iterator<Any> seqElements = seq.iterator();
    while (seqElements.hasNext()) {
      final Any any = seqElements.next();
      writeAny(writer, any);
    }
    writer.writeSequenceEnd();
  }

  /**
   * helper method to write an (Any) Value.
   */
  private void writeValue(final IpcStreamWriter writer, final Value value) throws IOException {
    if (value != null) {
      if (value.isBoolean()) {
        writer.writeScalarBoolean(value.asBoolean());
      } else if (value.isLong()) {
        writer.writeScalarLong(value.asLong());
      } else if (value.isDouble()) {
        writer.writeScalarDouble(value.asDouble());
      } else {
        // DATE and DATETIME are not directly supported, so String is used for them, too.
        writer.writeScalarString(value.asString());
      }
    }
  }

}
