/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.http.server.internal;

import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.http.server.HttpService;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.conversion.DataSizeParser;

/** loads configuration file <tt>httpserver.properties</tt>. */
public class HttpServerProperties {

  /** configuration filename. */
  private static final String FILENAME = "httpserver.properties";

  /** the loaded properties. */
  private final Properties _properties = new Properties();

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** create instance and load properties. */
  public HttpServerProperties() {
    loadConfiguration();
  }

  /** load configuration. */
  public void loadConfiguration() {
    try {
      final InputStream propertyFile = ConfigUtils.getConfigStream(HttpService.BUNDLE_NAME, FILENAME);
      try {
        _properties.load(propertyFile);
      } catch (final Exception ex) {
        _log.warn("Failed to load configuration " + FILENAME + " for bundle " + HttpService.BUNDLE_NAME, ex);
      } finally {
        IOUtils.closeQuietly(propertyFile);
      }
    } catch (final Exception ex) {
      _log.info("No configuration " + FILENAME + " found for bundle " + HttpService.BUNDLE_NAME, ex);
    }
  }

  /** get value for key, if set. */
  public String getProperty(final String key) {
    return _properties.getProperty(key);
  }

  /** get value for key, if set, else return defaultValue. */
  public String getProperty(final String key, final String defaultValue) {
    return _properties.getProperty(key, defaultValue);
  }

  /** parse a property value as a data size (512m, 2g, ...). */
  public long getDataSize(final String key, final String defaultValue) {
    try {
      return DataSizeParser.parse(getProperty(key, defaultValue));
    } catch (final Exception ex) {
      _log.info("Failed to parse configured value for " + key, ex);
      return DataSizeParser.parse(defaultValue);
    }
  }
}
