/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.test;

import java.io.File;

import junit.framework.TestCase;

import org.apache.commons.io.FileUtils;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.file.FileToRecordConverter;

public class TestFileToRecordConverter extends TestCase {

  private final FileToRecordConverter _converter = new FileToRecordConverter(DataFactory.DEFAULT);

  private File _rootFolder;

  @Override
  public void setUp() throws Exception {
    _rootFolder = FileystemHelper.initTestFilesystem();
  }

  @Override
  public void tearDown() throws Exception {
    if (_rootFolder != null) {
      FileUtils.deleteDirectory(_rootFolder);
    }
  }

  public void testFileToRecordWithoutAttachment() throws Exception {
    final Record record = _converter.fileToRecord(new File(_rootFolder, "d.e"), "testDataSource", false);
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertEquals("d.e", record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_NAME));
    assertEquals("e", record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_EXTENSION));
    assertEquals(_rootFolder.getPath(),
      record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_FOLDER));
    assertEquals(0, record.getMetadata().getLongValue(FileToRecordConverter.PROPERTY_FILE_SIZE).longValue());
    assertTrue(record.getMetadata().containsKey(FileToRecordConverter.PROPERTY_FILE_PATH));
    assertTrue(record.getMetadata().containsKey(FileToRecordConverter.PROPERTY_FILE_LAST_MODIFIED));
    assertTrue(record.getMetadata().containsKey(ImportingConstants.ATTRIBUTE_DELTA_HASH));
    assertFalse(record.hasAttachment(FileToRecordConverter.ATTACHMENT_FILE_CONTENT));
  }

  public void testFileToRecordWithAttachment() throws Exception {
    final Record record = _converter.fileToRecord(new File(_rootFolder, "d.e"), "testDataSource", true);
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertEquals("d.e", record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_NAME));
    assertEquals("e", record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_EXTENSION));
    assertEquals(_rootFolder.getPath(),
      record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_FOLDER));
    assertEquals(0, record.getMetadata().getLongValue(FileToRecordConverter.PROPERTY_FILE_SIZE).longValue());
    assertTrue(record.getMetadata().containsKey(FileToRecordConverter.PROPERTY_FILE_PATH));
    assertTrue(record.getMetadata().containsKey(FileToRecordConverter.PROPERTY_FILE_LAST_MODIFIED));
    assertTrue(record.getMetadata().containsKey(ImportingConstants.ATTRIBUTE_DELTA_HASH));
    assertTrue(record.hasAttachment(FileToRecordConverter.ATTACHMENT_FILE_CONTENT));
  }

  public void testFileToRecordWithoutExtension() throws Exception {
    final Record record = _converter.fileToRecord(new File(_rootFolder, "c"), "testDataSource", true);
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertEquals("c", record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_NAME));
    assertEquals("", record.getMetadata().getStringValue(FileToRecordConverter.PROPERTY_FILE_EXTENSION));
  }

  public void testFileToRecordOnDir() throws Exception {
    try {
      _converter.fileToRecord(new File(_rootFolder, "/a"), "testDataSource", false);
      fail("expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      assertNotNull(e);
    }
  }

  public void testDirectoryToRecord() throws Exception {
    final Record record = _converter.dirToRecord(new File(_rootFolder, "a"), "testDataSource");
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertTrue(record.getMetadata().containsKey(FileToRecordConverter.PROPERTY_FILE_FOLDER));
  }

  public void testDirectoryToRecordOnFile() throws Exception {
    try {
      _converter.dirToRecord(new File(_rootFolder, "c"), "testDataSource");
      fail("expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      assertNotNull(e);
    }
  }

  public void testAddAttachment() throws Exception {
    final Record record = _converter.fileToRecord(new File(_rootFolder, "c"), "testDataSource", false);
    assertNotNull(record);
    assertFalse(record.hasAttachment(FileToRecordConverter.ATTACHMENT_FILE_CONTENT));
    _converter.addAttachment(record);
    assertTrue(record.hasAttachment(FileToRecordConverter.ATTACHMENT_FILE_CONTENT));
  }

  public void testAddAttachmentWithoutPath() throws Exception {
    final Record record = _converter.fileToRecord(new File(_rootFolder, "/c"), "testDataSource", false);
    assertNotNull(record);
    assertFalse(record.hasAttachment(FileToRecordConverter.ATTACHMENT_FILE_CONTENT));
    record.getMetadata().remove(FileToRecordConverter.PROPERTY_FILE_PATH);
    try {
      _converter.addAttachment(record);
      fail("expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      assertNotNull(e);
    }
  }

}
