/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.importing.test;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.file.FileToRecordConverter;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;

public class TestFileCrawling extends AImportingIntegrationTest {

  private static final String JOBNAME_CRAWLFILES = "crawlFiles";

  public void testInitialCrawl() throws Exception {
    crawlFilesFromConfig("files10", "testInitialCrawl");
    assertTrue(_deltaService.getSourceIds().contains("files"));
    final int expectedRecordCount = 10;
    checkAddedBulks(expectedRecordCount);
  }

  public void testCrawlMultipleLevels() throws Exception {
    crawlFilesFromConfig("filesMultiLevel", "testCrawlMultipleLevels");
    assertTrue(_deltaService.getSourceIds().contains("files"));
    final int expectedRecordCount = 31;
    checkAddedBulks(expectedRecordCount);
  }

  public void testNoUpdatesCrawl() throws Exception {
    final File crawlDir = crawlFilesFromConfig("files10", "testNoUpdatesCrawl");
    checkAddedBulks(10);
    _objectStore.clearStore(STORENAME_BULKS);
    crawlFiles(crawlDir);
    checkAddedBulks(0);
  }

  public void testCrawlAdditionalFiles() throws Exception {
    crawlFilesFromConfig("files59", "testCrawlAdditionalFiles");
    checkAddedBulks(59);
    _objectStore.clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", "testCrawlAdditionalFiles");
    checkAddedBulks(10);
  }

  public void testCrawlUpdates() throws Exception {
    final File workspaceDir = copyConfigFilesToWorkspace("files10", "testCrawlUpdates");
    crawlFiles(workspaceDir);
    checkAddedBulks(10);
    _objectStore.clearStore(STORENAME_BULKS);
    // modify half of the files
    final File[] files = workspaceDir.listFiles();
    for (int i = 0; i < files.length; i++) {
      if (i % 2 == 0) {
        files[i].setLastModified(System.currentTimeMillis());
      }
    }
    crawlFiles(workspaceDir);
    checkUpdateBulks(5);
  }

  /** copy a directory from configuration to workspace, skip .svn directories. */
  private File copyConfigFilesToWorkspace(final String configDirName, final String workspaceDirName)
    throws IOException {
    final File configDir = ConfigUtils.getConfigFile(AllTests.BUNDLE_ID, configDirName);
    final File workspaceDir = WorkspaceHelper.createWorkingDir(AllTests.BUNDLE_ID, workspaceDirName);
    FileUtils
      .copyDirectory(configDir, workspaceDir, FileFilterUtils.makeSVNAware(FileFilterUtils.trueFileFilter()));
    return workspaceDir;
  }

  private File crawlFilesFromConfig(final String configDirName, final String workspaceDirName) throws Exception {
    final File workspaceDir = copyConfigFilesToWorkspace(configDirName, workspaceDirName);
    crawlFiles(workspaceDir);
    return workspaceDir;
  }

  private void crawlFiles(final File dirToCrawl) throws Exception, BulkbuilderException {
    final String crawlJobId = startFileCrawlerJob(dirToCrawl);
    waitForJobRunCompleted(JOBNAME_CRAWLFILES, crawlJobId, 15000);
    _bulkbuilder.commitJob(JOBNAME_BUILDBULKS);
  }

  private void checkAddedBulks(final int expectedRecordCount) throws ObjectStoreException, Exception {
    final Collection<StoreObject> bulks = _objectStore.getStoreObjectInfos(STORENAME_BULKS, BUCKET_ADDED);
    assertNotNull(bulks);
    if (expectedRecordCount == 0) {
      assertTrue(bulks.isEmpty());
    } else {
      assertEquals(expectedRecordCount, checkRecords(bulks, false));
    }
  }

  private void checkUpdateBulks(final int expectedRecordCount) throws ObjectStoreException, Exception {
    final Collection<StoreObject> bulks = _objectStore.getStoreObjectInfos(STORENAME_BULKS, BUCKET_ADDED);
    assertNotNull(bulks);
    assertEquals(expectedRecordCount, checkRecords(bulks, true));
  }

  private int checkRecords(final Collection<StoreObject> bulks, final boolean update) throws Exception {
    int recordCount = 0;
    for (final StoreObject bulk : bulks) {
      final InputStream bulkStream = _objectStore.readObject(STORENAME_BULKS, bulk.getId());
      try {
        final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream);
        while (records.hasNext()) {
          final Record record = records.next();
          assertNotNull(record);
          recordCount++;
          assertNotNull(record.getId());
          assertEquals("files", record.getSource());
          final AnyMap metadata = record.getMetadata();
          if (update) {
            assertTrue(metadata.getBooleanValue(ImportingConstants.ATTRIBUTE_UPDATE));
          } else {
            assertFalse(metadata.containsKey(ImportingConstants.ATTRIBUTE_UPDATE));
          }
          assertTrue(metadata.containsKey(ImportingConstants.ATTRIBUTE_DELTA_HASH));
          assertTrue(metadata.containsKey(FileToRecordConverter.PROPERTY_FILE_NAME));
          assertTrue(metadata.containsKey(FileToRecordConverter.PROPERTY_FILE_PATH));
          assertTrue(metadata.containsKey(FileToRecordConverter.PROPERTY_FILE_FOLDER));
          assertTrue(metadata.get(FileToRecordConverter.PROPERTY_FILE_LAST_MODIFIED).isDateTime());
          assertTrue(metadata.get(FileToRecordConverter.PROPERTY_FILE_SIZE).isLong());
          assertTrue(record.hasAttachment(FileToRecordConverter.ATTACHMENT_FILE_CONTENT));
        }
      } finally {
        IOUtils.closeQuietly(bulkStream);
      }
    }
    return recordCount;
  }

  private String startFileCrawlerJob(final File dirToCrawl) throws Exception {
    final JobDefinition jobTemplate = _defPersistence.getJob(JOBNAME_CRAWLFILES + "Template");
    final AnyMap jobAny = jobTemplate.toAny(false);
    jobAny.put("name", JOBNAME_CRAWLFILES);
    jobAny.getMap("parameters").put("rootFolder", dirToCrawl.getAbsolutePath());
    final JobDefinition job = new JobDefinition(jobAny);
    _defPersistence.addJob(job);
    return _jobRunEngine.startJob(JOBNAME_CRAWLFILES, JobRunMode.RUNONCE);
  }
}
