/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.Outputs;
import org.eclipse.smila.taskworker.output.RecordOutput;

/**
 * Worker implementation that performs delta checks.
 * 
 * @author stuc07
 * 
 */
public class DeltaCheckerWorker implements Worker {

  private static final String WORKER_NAME = "deltaChecker";

  private static final String INPUT_SLOT_NAME = "recordsToCheck";

  private static final String OUTPUT_SLOT_NAME = "updatedRecords";

  private final Log _log = LogFactory.getLog(getClass());

  private DeltaService _deltaService;

  @Override
  public String getName() {
    return WORKER_NAME;
  }

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    final String jobRunId = taskContext.getTask().getProperties().get(Task.PROPERTY_JOB_RUN_ID);
    if (jobRunId == null) {
      throw new IllegalArgumentException("Missing property '" + Task.PROPERTY_JOB_RUN_ID + "' in task: "
        + taskContext.getTask());
    }
    final Inputs inputs = taskContext.getInputs();
    final RecordInput recordInput = inputs.getAsRecordInput(INPUT_SLOT_NAME);
    final Outputs outputs = taskContext.getOutputs();
    final RecordOutput recordOutput = outputs.getAsRecordOutput(OUTPUT_SLOT_NAME);
    Record record = recordInput.getRecord();
    while (record != null) {
      final Record updatedRecord = checkRecord(taskContext, jobRunId, record);
      if (updatedRecord != null) {
        recordOutput.writeRecord(updatedRecord);
      }
      record = recordInput.getRecord();
    }
  }

  /** get delta hash value from record and check with DeltaService. */
  private Record checkRecord(final TaskContext taskContext, final String jobRunId, final Record record)
    throws ObjectStoreException, IOException {
    final String deltaHash = record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_DELTA_HASH);
    if (deltaHash == null) {
      if (_log.isDebugEnabled()) {
        _log.debug("Record " + record.getId() + " has no hash attribute value, is added to output.");
      }
      return record;
    } else {
      try {
        final long startDelta = taskContext.getTimestamp();
        final State deltaStatus = _deltaService.checkState(record.getSource(), record.getId(), jobRunId, deltaHash);
        taskContext.measureTime("invokeDeltaService", startDelta);
        if (deltaStatus != State.UPTODATE) {
          if (_log.isDebugEnabled()) {
            _log.debug("Record " + record.getId() + " will be added to output, has status " + deltaStatus);
          }
          if (deltaStatus == State.CHANGED) {
            record.getMetadata().put(ImportingConstants.ATTRIBUTE_UPDATE, true);
          }
          return record;
        }
      } catch (final Exception e) {
        taskContext.getLog().error("Error checking record " + record.getId(), e);
      }
    }
    _log.debug("Record " + record.getId() + " is up to date");
    return null;
  }

  public void setDeltaService(final DeltaService service) {
    _deltaService = service;
  }

  public void unsetDeltaService(final DeltaService service) {
    if (_deltaService == service) {
      _deltaService = null;
    }
  }

}
