/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Date;

import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.httphandler.BucketHandler;
import org.eclipse.smila.jobmanager.httphandler.BucketsHandler;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test cases for the {@link BucketHandler} and {@link BucketsHandler}.
 */
public class TestBucketHandler extends DeclarativeServiceTestCase {

  private BucketHandler _bucketHandler;

  private BucketsHandler _bucketsHandler;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    RequestHandler service = getService(RequestHandler.class, "(uriPattern=/jobmanager/buckets/\\([^/]+\\)/?$)");
    assertTrue(service instanceof BucketHandler);
    _bucketHandler = (BucketHandler) service;
    service = getService(RequestHandler.class, "(uriPattern=/jobmanager/buckets/?$)");
    assertTrue(service instanceof BucketsHandler);
    _bucketsHandler = (BucketsHandler) service;
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * tests adding, getting and deleting a definition via the request handler.
   * 
   * @throws Exception
   */
  public void testAddAndGetAndDeleteDefinition() throws Exception {
    final String jobBaseUrlPart = _bucketHandler.getRootContextPath() + "/jobmanager/buckets/";
    final String definitionName = "testAddAndGetAndDeleteDefinition";
    final Record definition = DataFactory.DEFAULT.createRecord();
    definition.getMetadata().put("name", definitionName);
    definition.getMetadata().put("type", "recordBulks");
    definition.getMetadata().put("additionalData", "some additional data.");
    // this must be removed when inserted via handler.
    definition.getMetadata().put(DefinitionBase.KEY_READ_ONLY, true);

    // create a definition
    final Date startDate = new Date();
    final AnyMap result = (AnyMap) _bucketsHandler.process("POST", jobBaseUrlPart, definition);
    assertEquals(definition.getMetadata().get("name").toString(), result.get("name").toString());
    // get the complete definition
    final Record returnDetails = DataFactory.DEFAULT.createRecord();
    returnDetails.getMetadata().put(JobManagerConstants.KEY_RETURN_DETAILS, true);
    AnyMap retrievedDefinition =
      (AnyMap) _bucketHandler.process("GET", jobBaseUrlPart + definitionName, returnDetails);
    final Date endDate = new Date();
    // check the timestamp of the result.
    final Date dateFromDefinition = retrievedDefinition.getDateTimeValue(DefinitionBase.KEY_TIMESTAMP);
    assertEquals(result.getDateTimeValue(DefinitionBase.KEY_TIMESTAMP), dateFromDefinition);
    assertTrue(startDate.compareTo(dateFromDefinition) <= 0);
    assertTrue(endDate.compareTo(dateFromDefinition) >= 0);
    // be sure that the readonly flag has been removed from the input any.
    assertFalse(retrievedDefinition.containsKey(DefinitionBase.KEY_READ_ONLY));
    // we must remove timestamp and readOnly flag for comparison...
    final Any timestampAny = definition.getMetadata().get(JobManagerConstants.TIMESTAMP);
    assertEquals(timestampAny.asValue().asDateTime(), dateFromDefinition);
    definition.getMetadata().remove(DefinitionBase.KEY_TIMESTAMP);
    definition.getMetadata().remove(DefinitionBase.KEY_READ_ONLY);
    retrievedDefinition.remove(JobManagerConstants.TIMESTAMP);
    assertEquals(definition.getMetadata(), retrievedDefinition);
    // get only the relevant part
    retrievedDefinition = (AnyMap) _bucketHandler.process("GET", jobBaseUrlPart + definitionName, null);
    final AnyMap reducedAny = DataFactory.DEFAULT.cloneAnyMap(definition.getMetadata());
    reducedAny.remove("additionalData");
    // re-set the timestamp to be able to check it (we removed it above)
    reducedAny.put(DefinitionBase.KEY_TIMESTAMP, timestampAny);
    assertEquals(reducedAny, retrievedDefinition);
    // remove the workflow
    _bucketHandler.process("DELETE", jobBaseUrlPart + definitionName, null);

    // check if it really got removed...
    try {
      _bucketHandler.process("GET", jobBaseUrlPart + definitionName, null);
      fail("ConfigNotFoundException expected.");
    } catch (final ConfigNotFoundException e) {
      ;// ignore
    }
  }

  /** check if a bucket with an invalid name is rejected. */
  public void testAddBucketWithInvalidName() throws Exception {
    try {
      final Record bucket = DataFactory.DEFAULT.createRecord();
      bucket.getMetadata().put("name", "this/name+is=invalid?");
      _bucketsHandler.process("POST", "jobmanager/buckets", bucket);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK
    }
  }
}
