/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.events.JobEvent;
import org.eclipse.smila.jobmanager.events.JobListener;
import org.eclipse.smila.jobmanager.events.PrepareToFinishEvent;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.osgi.framework.Constants;

/**
 * Tests {@link JobListener} notification.
 */
public class TestJobListener extends JobManagerTestBase {

  /**
   * Dummy JobEvent Listener.
   * 
   */
  private static class JobEventListenerDummy implements JobListener {

    /** the events that have been processed. */
    private final List<JobEvent> _processedEvents = new ArrayList<JobEvent>();

    /**
     * {@inheritDoc}
     */
    @Override
    public void processJobEvent(final JobEvent evt) {
      _processedEvents.add(evt);
    }

    /**
     * @return the processed events
     */
    public List<JobEvent> getProcessedEvents() {
      return _processedEvents;
    }

    /**
     * clears the processed event list.
     */
    public void clearProcessedEvents() {
      _processedEvents.clear();
    }
  }

  /**
   * Tests if adding and removing a job event listener works.
   * 
   * @throws Exception
   *           unhandled exception
   */
  public void testAddAndRemoveJobListener() throws Exception {
    final JobEventListenerDummy jobEventListener = new JobEventListenerDummy();
    final String jobName = "testAddAndRemoveJobListener";
    final String workflow = "testWorkflow";
    final String workerParameter = "test";
    final String tempStoreName = STORE_NAME;
    final String storeName = STORE_NAME;
    final AnyMap parameters = AccessAny.FACTORY.createAnyMap();
    parameters.put("tempStore", tempStoreName);
    parameters.put("store", storeName);
    parameters.put("workerParameter", workerParameter);

    final Dictionary<Object, Object> dictionary = new Hashtable<Object, Object>();
    // make this listener the top prio service...
    dictionary.put(Constants.SERVICE_RANKING, Integer.MAX_VALUE);
    final JobListener jobListener = registerService(jobEventListener, null, JobListener.class, 1000);
    assertEquals(jobEventListener, jobListener);
    assertNotNull(jobListener);

    // add a Job
    addJob(jobName, workflow, parameters);
    assertEquals(0, jobEventListener.getProcessedEvents().size());

    // start the job
    final String jobRunId = startJob(jobName);
    assertEquals(0, jobEventListener.getProcessedEvents().size());

    // try to finish a wrong job run
    /*try {
      _jobManager.finishJob(jobName, "notExistingJobRun");
      fail("must not be able to finish wrong job run.");
    } catch (final JobManagerException je) {
      assertEquals(0, jobEventListener.getProcessedEvents().size());
    }*/

    // finish the job
    _jobRunEngine.finishJob(jobName, jobRunId);
    assertEquals(1, jobEventListener.getProcessedEvents().size());
    assertEquals(PrepareToFinishEvent.class, jobEventListener.getProcessedEvents().get(0).getClass());

    // try to finish it a second time
    try {
      _jobRunEngine.finishJob(jobName, jobRunId);
      fail("must not be able to finish job twice.");
    } catch (final JobManagerException je) {
      assertEquals(1, jobEventListener.getProcessedEvents().size());
    }

    jobEventListener.clearProcessedEvents();

    // unregister the jobListener
    unregisterService(JobListener.class);

    // and start and finish the job again...
    // start the job
    final String secondJobRunId = startJob(jobName);
    // finish the job
    _jobRunEngine.finishJob(jobName, secondJobRunId);
    // since we've unregistered, there must not be any event handling...
    assertEquals(0, jobEventListener.getProcessedEvents().size());
  }
}
