/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Schank (both Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.httphandler.WorkerHandler;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test cases for the {@link WorkerHandler}.
 */
public class TestWorkerHandler extends DeclarativeServiceTestCase {

  private WorkerHandler _workerHandler;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    final RequestHandler service =
      getService(RequestHandler.class, "(uriPattern=/jobmanager/workers/\\([^/]+\\)/?$)");
    assertTrue(service instanceof WorkerHandler);
    _workerHandler = (WorkerHandler) service;
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * tests getting a definition via the request handler.
   * 
   * @throws Exception
   */
  public void testGetDefinition() throws Exception {
    final String jobBaseUrlPart = _workerHandler.getRootContextPath() + "/jobmanager/workers/";
    final String definitionName = "testOptionalParametersWorker";
    // get the complete definition
    final Record returnDetails = DataFactory.DEFAULT.createRecord();
    returnDetails.getMetadata().put(JobManagerConstants.KEY_RETURN_DETAILS, true);
    AnyMap retrievedDefinition =
      (AnyMap) _workerHandler.process("GET", jobBaseUrlPart + definitionName, returnDetails);
    // check the comments field.
    assertEquals("my comment", retrievedDefinition.getStringValue("comment"));
    // check additional info in parameter.
    AnySeq paramSeq = retrievedDefinition.getSeq("parameters");
    for (final Any param : paramSeq) {
      assertTrue(param.isMap());
      if (param.asMap().getStringValue("name").equals("optionalParameter")) {
        assertEquals("blabla", param.asMap().getMap("additionalInfo").getStringValue("add1"));
        assertEquals("blub", param.asMap().getMap("additionalInfo").getStringValue("add2"));
      } else if (param.asMap().getStringValue("name").equals("workerParameter")) {
        assertEquals(2, param.asMap().size());
        assertTrue(param.asMap().containsKey("comment"));
      } else {
        fail("Unexpected parameter found: " + param);
      }
    }
    // get only the relevant part
    retrievedDefinition = (AnyMap) _workerHandler.process("GET", jobBaseUrlPart + definitionName, null);
    // check absence of additional info in parameter.
    paramSeq = retrievedDefinition.getSeq("parameters");
    for (final Any param : paramSeq) {
      assertTrue(param.isMap());
      if (param.asMap().getStringValue("name").equals("optionalParameter")) {
        assertEquals(2, param.asMap().size());
        assertEquals(Boolean.TRUE, param.asMap().getBooleanValue("optional"));
      } else if (param.asMap().getStringValue("name").equals("workerParameter")) {
        assertEquals(1, param.asMap().size());
      } else {
        fail("Unexpected parameter found: " + param);
      }
    }
    assertFalse(retrievedDefinition.containsKey("comment"));
  }

  /**
   * check if unspecified parts of parameter definition are reported correctly.
   * 
   * @throws Exception
   */
  public void testGetWorkerDefinition() throws Exception {
    final String workerName = JobManagerTestBase.WORKER_OPTIONAL_PARAM;
    final String workerBaseUrlPart = _workerHandler.getRootContextPath() + "/jobmanager/workers/";
    final Record returnDetails = DataFactory.DEFAULT.createRecord();
    returnDetails.getMetadata().put(JobManagerConstants.KEY_RETURN_DETAILS, true);
    final AnyMap workerAny = (AnyMap) _workerHandler.process("GET", workerBaseUrlPart + workerName, returnDetails);
    assertEquals(workerName, workerAny.getStringValue("name"));
    assertEquals(2, workerAny.getSeq("modes").size());
    final AnySeq parameters = workerAny.getSeq("parameters");
    assertEquals(2, parameters.size());
    assertEquals("workerParameter", parameters.getMap(0).getStringValue("name"));
    assertEquals("a mandatory parameter", parameters.getMap(0).getStringValue("comment"));
    assertEquals("optionalParameter", parameters.getMap(1).getStringValue("name"));
    assertEquals(true, parameters.getMap(1).getBooleanValue("optional").booleanValue());
    assertEquals("an optional parameter", parameters.getMap(1).getStringValue("comment"));
    assertTrue(parameters.getMap(1).get("additionalInfo").isMap());
    assertEquals("blabla", parameters.getMap(1).getMap("additionalInfo").getStringValue("add1"));
    assertEquals("blub", parameters.getMap(1).getMap("additionalInfo").getStringValue("add2"));
    assertFalse(workerAny.containsKey("input"));
    assertEquals(1, workerAny.getSeq("output").size());
  }

}
