/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import java.util.Collection;

import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;

/** Access and modify job manager definitions. */
public interface DefinitionPersistence {

  /**
   * remove a bucket from the storage.
   * 
   * @param name
   *          bucket name.
   * @throws PersistenceException
   *           error deleting definition.
   * @throws InvalidConfigException
   *           error deleting definition due to dependency constraints
   */
  void removeBucket(final String name) throws PersistenceException, InvalidConfigException;

  /**
   * @return all data object type names.
   */
  Collection<String> getDataObjectTypes();

  /**
   * @param name
   *          the name of the data object type definition
   * @return the data object type definition for the given name, or <code>null</code> if not found
   */
  DataObjectTypeDefinition getDataObjectType(final String name);

  /**
   * @return all worker names.
   */
  Collection<String> getWorkers();

  /**
   * @param name
   *          the name of the worker definition
   * @return returns the worker definition with the given name, or <code>null</code> if not found
   */
  WorkerDefinition getWorker(final String name);

  /**
   * @return names of preconfigured and user-defined buckets.
   * @throws PersistenceException
   *           error reading from storage.
   */
  Collection<String> getBuckets() throws PersistenceException;

  /**
   * get a bucket from configuration or storage.
   * 
   * @param name
   *          bucket name
   * @return bucket definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  BucketDefinition getBucket(final String name) throws PersistenceException;

  /**
   * add one bucket to definition storage. No further validation.
   * 
   * @param bucketDefinition
   *          new bucket.
   * @throws PersistenceException
   *           error writing to storage.
   * @throws InvalidConfigException
   *           error adding definition due to dependency constraints.
   */
  void addBucket(final BucketDefinition bucketDefinition) throws PersistenceException, InvalidConfigException;

  /**
   * add a workflow to definition storage. No further validation.
   * 
   * @param workflowDefinition
   *          new workflow.
   * @throws PersistenceException
   *           error writing to storage.
   * @throws InvalidConfigException
   *           the configuration of the workflow is incorrect (e.g. would lead to an invalid store name.)
   */
  void addWorkflow(final WorkflowDefinition workflowDefinition) throws PersistenceException, InvalidConfigException;

  /**
   * remove a workflow from the storage.
   * 
   * @param name
   *          workflow name.
   * @throws PersistenceException
   *           error deleting definition
   * @throws InvalidConfigException
   *           error deleting definition due to dependency constraints
   */
  void removeWorkflow(final String name) throws PersistenceException, InvalidConfigException;

  /**
   * get a workflow definition from storage (try first) or configuration (if not found in storage).
   * 
   * @param name
   *          workflow name
   * @return workflow definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  WorkflowDefinition getWorkflow(final String name) throws PersistenceException;

  /**
   * @return (unique) names of defined workflows.
   * @throws PersistenceException
   *           error reading from storage
   */
  Collection<String> getWorkflows() throws PersistenceException;

  /**
   * add a job to definition storage. No further validation.
   * 
   * @param jobDefinition
   *          new job.
   * @throws PersistenceException
   *           error writing to storage.
   * @throws InvalidConfigException
   *           the job is misconfigured (e.g. has invalid parameter settings or not enough parameters to satisfy data
   *           object type definitions or worker parameters).
   */
  void addJob(final JobDefinition jobDefinition) throws PersistenceException, InvalidConfigException;

  /**
   * remove a job from the storage.
   * 
   * @param name
   *          job name.
   * @throws PersistenceException
   *           error deleting definition
   * @throws IllegalJobStateException
   *           the job is in an illegal state (i.e. it is running and cannot be removed).
   */
  void removeJob(final String name) throws PersistenceException, IllegalJobStateException;

  /**
   * @param jobName
   *          job name
   * @return true if a job with this name is defined
   * @throws PersistenceException
   */
  boolean hasJob(final String jobName) throws PersistenceException;

  /**
   * get a job name from configuration or storage.
   * 
   * @param name
   *          job name
   * @return job definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  JobDefinition getJob(final String name) throws PersistenceException;

  /**
   * @return names of defined jobs.
   * @throws PersistenceException
   *           error reading from storage
   */
  Collection<String> getJobs() throws PersistenceException;
}
