/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.Date;

import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;

/**
 * Implements the handling of HTTP requests to request the workflow list or to define new workflows. <br>
 * 
 * URL pattern: <code>smila/jobmanager/workflows/</code> <br>
 * 
 * Methods allowed: <code>GET, POST</code>
 */
public class WorkflowsHandler extends AJobManagerHandler {
  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final DefinitionPersistence defPersistence = getDefinitionPersistence();
    final AnyMap resultMap = DataFactory.DEFAULT.createAnyMap();
    if (isGetRequest(method)) {
      final AnySeq workflowList = DataFactory.DEFAULT.createAnySeq();
      boolean returnDetails = false;
      if (inputRecord != null && inputRecord.getMetadata().containsKey(JobManagerConstants.KEY_RETURN_DETAILS)) {
        returnDetails =
          inputRecord.getMetadata().getBooleanValue(JobManagerConstants.KEY_RETURN_DETAILS).booleanValue();
      }
      for (final String workflowName : defPersistence.getWorkflows()) {
        final AnyMap map = DataFactory.DEFAULT.createAnyMap();
        map.put("name", workflowName);
        map.put("url", URLCreator.create(getRequestHost(), requestUri, workflowName).toExternalForm());
        if (returnDetails && defPersistence.getWorkflow(workflowName).getTimestamp() != null) {
          map.put(JobManagerConstants.TIMESTAMP,
            DataFactory.DEFAULT.createDateTimeValue(defPersistence.getWorkflow(workflowName).getTimestamp()));
        }
        workflowList.add(map);
      }
      resultMap.put(WorkflowDefinition.KEY_WORKFLOWS, workflowList);
    }
    if (isPostRequest(method)) { // POST
      if (inputRecord == null) {
        throw new IllegalArgumentException("Missing input, cannot create workflows from nothing.");
      }
      final AnyMap definition = inputRecord.getMetadata();
      // create timestamp for this workflow and store it with the data
      final Any timestamp = inputRecord.getFactory().createDateTimeValue(new Date());
      definition.put(DefinitionBase.KEY_TIMESTAMP, timestamp);
      // remove readOnly flag if present.
      definition.remove(DefinitionBase.KEY_READ_ONLY);
      final WorkflowDefinition createdWorkflow = WorkflowDefinition.parseWorkflow(definition);
      defPersistence.addWorkflow(createdWorkflow);
      resultMap.put("name", createdWorkflow.getName());
      resultMap.put(JobManagerConstants.TIMESTAMP, timestamp);
      resultMap.put("url", URLCreator.create(getRequestHost(), requestUri, createdWorkflow.getName())
        .toExternalForm());
    }
    return resultMap;
  }

  /**
   * result status for POST requests should be CREATED.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if (isPostRequest(method)) {
      return HttpStatus.CREATED;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isGetOrPostRequest(method);
  }
}
