/*******************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Juergen Schumacher (empolis GmbH) - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.processing.bpel.test;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.processing.bpel.internal.ZkUpdateWatcher;
import org.eclipse.smila.zookeeper.ZooKeeperService;

/**
 * Test some basic operations of the workflow processor, not related to actual workflow processing.
 * 
 * @author jschumacher
 * 
 */
public class TestWorkflowUpdateWatcher extends AWorkflowProcessorTest {

  /** workflow updates are sent to this real processor. */
  private WorkflowProcessor _processor;

  /** another watcher to detect the notifications. */
  private ZkUpdateWatcher _watcher;

  /** mock processor to check if notifications were received. */
  private WorkflowProcessor _notifiedProcessor;

  /**
   * processor mockup to receive notifications from updater.
   */
  private class MockProcessor implements WorkflowProcessor {

    private final List<String> _workflows = new ArrayList<String>();

    @Override
    public String[] process(final String workflowName, final Blackboard blackboard, final String[] recordIds)
      throws ProcessingException {
      return recordIds;
    }

    @Override
    public List<String> getWorkflowNames() {
      return _workflows;
    }

    @Override
    public AnyMap getWorkflowDefinition(final String workflowName) throws ProcessingException {
      return null; // not needed
    }

    @Override
    public void setWorkflowDefinition(final String workflowName, final AnyMap workflowDefinition)
      throws ProcessingException {
      // nothing to do
    }

    @Override
    public void deleteWorkflowDefinition(final String workflowName) throws ProcessingException {
      // nothing to do
    }

    @Override
    public void synchronizeWorkflowDefinition(final String workflowName, final boolean isDeleted)
      throws ProcessingException {
      if (isDeleted) {
        _workflows.removeAll(Arrays.asList(workflowName));
      } else {
        _workflows.add(workflowName);
      }
    }
  }

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _processor = getService(WorkflowProcessor.class);
    assertNotNull("no WorkflowProcessor service found.", _processor);
    final ZooKeeperService zkService = getService(ZooKeeperService.class);
    assertNotNull(zkService);
    _notifiedProcessor = new MockProcessor();
    _watcher = new ZkUpdateWatcher();
    _watcher.setZkService(zkService);
    _watcher.registerProcessor(_notifiedProcessor);
    _watcher.initialize();
  }

  @Override
  protected void tearDown() throws Exception {
    _watcher.stopPolling();
    _watcher.stopWatching();
    super.tearDown();
  }

  @Override
  protected String getPipelineName() {
    return "EchoPipeline";
  }

  /** test creating a new workflow. */
  public void testNotifySetWorkflow() throws Exception {
    final String pipelineName = "testNotifySetWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    assertNull(_processor.getWorkflowDefinition(pipelineName));
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    _watcher.checkWorkflowVersions();
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
  }

  /** test updating a new workflow. */
  public void testNotifyUpdateWorkflow() throws Exception {
    final String pipelineName = "testNotifyUpdateWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    _watcher.checkWorkflowVersions();
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    final String updateBpel = copyLocalHelloWorldPipeline(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, updateBpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    _watcher.checkWorkflowVersions();
    assertEquals(2, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
  }

  /** test deleting a workflow. */
  public void testNotifyDeleteWorkflow() throws Exception {
    final String pipelineName = "testNotifyDeleteWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    _watcher.checkWorkflowVersions();
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _processor.deleteWorkflowDefinition(pipelineName);
    assertFalse(_processor.getWorkflowNames().contains(pipelineName));
    _watcher.checkWorkflowVersions();
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
  }

  /** test updating a new workflow. */
  public void testNotifyUpdateMultipleWorkflows() throws Exception {
    final String pipelineName1 = "testNotifyUpdateMultipleWorkflows1";
    final String pipelineName2 = "testNotifyUpdateMultipleWorkflows2";
    final String bpel1 = copyEchoPipeline(pipelineName1);
    final String bpel2 = copyLocalHelloWorldPipeline(pipelineName2);
    final AnyMap definition1 = createWorkflowDefinition(pipelineName1, bpel1);
    _processor.setWorkflowDefinition(pipelineName1, definition1);
    _watcher.checkWorkflowVersions();
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName1));
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName2));
    final AnyMap definition2 = createWorkflowDefinition(pipelineName1, bpel2);
    _processor.setWorkflowDefinition(pipelineName2, definition2);
    _watcher.checkWorkflowVersions();
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName1));
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName2));
    _processor.setWorkflowDefinition(pipelineName2, definition2);
    _watcher.checkWorkflowVersions();
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName1));
    assertEquals(2, count(_notifiedProcessor.getWorkflowNames(), pipelineName2));
    _processor.setWorkflowDefinition(pipelineName1, definition1);
    _watcher.checkWorkflowVersions();
    assertEquals(2, count(_notifiedProcessor.getWorkflowNames(), pipelineName1));
    assertEquals(2, count(_notifiedProcessor.getWorkflowNames(), pipelineName2));
    _processor.deleteWorkflowDefinition(pipelineName1);
    _watcher.checkWorkflowVersions();
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName1));
    assertEquals(2, count(_notifiedProcessor.getWorkflowNames(), pipelineName2));
    _processor.deleteWorkflowDefinition(pipelineName2);
    _watcher.checkWorkflowVersions();
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName1));
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName2));
  }

  /** test automatic polling. */
  public void testPollingForUpdates() throws Exception {
    final String pipelineName = "testPollingForUpdates";
    _watcher.startPolling(1);
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    Thread.sleep(1200);
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _processor.setWorkflowDefinition(pipelineName, definition);
    Thread.sleep(1200);
    assertEquals(2, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _processor.deleteWorkflowDefinition(pipelineName);
    Thread.sleep(1200);
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _watcher.stopPolling();
    _processor.setWorkflowDefinition(pipelineName, definition);
    Thread.sleep(1200);
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
  }

  /** test notification using ZK watches. */
  public void testWatchingForUpdates() throws Exception {
    final String pipelineName = "testWatchingForUpdates";
    _watcher.startWatching();
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    Thread.sleep(100);
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _processor.setWorkflowDefinition(pipelineName, definition);
    Thread.sleep(100);
    assertEquals(2, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _processor.deleteWorkflowDefinition(pipelineName);
    Thread.sleep(100);
    assertEquals(0, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _watcher.stopWatching();
    _processor.setWorkflowDefinition(pipelineName, definition);
    Thread.sleep(100);
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
    _processor.setWorkflowDefinition(pipelineName, definition);
    Thread.sleep(100);
    // this notification not received, becauase watch was stopped.
    assertEquals(1, count(_notifiedProcessor.getWorkflowNames(), pipelineName));
  }

  /** count occurrences of element in list. */
  private Object count(final Collection<?> list, final Object element) {
    int count = 0;
    for (final Object o : list) {
      if (o.equals(element)) {
        ++count;
      }
    }
    return count;
  }
}
