/*****************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation 
 *****************************************************************************************************************/

package org.eclipse.smila.processing.bpel.activities;

import org.apache.ode.bpel.common.FaultException;
import org.apache.ode.bpel.compiler.api.CompilationException;
import org.apache.ode.bpel.compiler.v2.CompilerContext;
import org.apache.ode.bpel.extension.ExtensibleElement;
import org.apache.ode.bpel.extension.ExtensionOperation;
import org.apache.ode.bpel.extension.ExtensionValidator;
import org.apache.ode.bpel.rtrep.common.extension.AbstractExtensionBundle;
import org.apache.ode.bpel.rtrep.common.extension.ExtensionContext;
import org.apache.ode.bpel.rtrep.v2.OActivity;
import org.apache.ode.bpel.rtrep.v2.OExtensionActivity;
import org.apache.ode.bpel.rtrep.v2.OProcess;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.bpel.util.BpelConstants;
import org.w3c.dom.Element;

/**
 * ODE extension bundle for compiling and invoking invokePipelet and invokeService extension activities.
 */
public class PipeletExtensionBundle extends AbstractExtensionBundle {

  /** for validating and parsing pipelet invocations from BPEL. */
  private static final PipeletActivityValidator VALIDATOR = new PipeletActivityValidator();

  @Override
  public String getNamespaceURI() {
    return BpelConstants.NAMESPACE_PROCESSOR;
  }

  @Override
  public void registerExtensionActivities() {
    registerExtensionOperation(BpelConstants.TAG_INVOKE_PIPELET, InvokePipeletActivity.class);
  }

  /** invoke a pipelet. */
  private static void invokeActivity(final ExtensionContext context, final Element element) {
    final OActivity activity = context.getOActivity();
    final String key = PipeletExtensionBundle.getActivityKey(activity);
    PipeletManager.getServiceInstance().invokeActivity(key, context, element);
  }

  /** validate a pipelet invocation and register it in the PipeletManager. */
  private static void registerActivity(final OExtensionActivity activity, final ExtensibleElement element,
    final CompilerContext compilerContext) {
    final OProcess process = activity.getOwner();
    final String key = getActivityKey(activity);
    final Element content = element.getNestedElement();
    final PipeletHolder pipeletHolder =
      VALIDATOR.validateActivity(process, activity, content, key, compilerContext);
    try {
      PipeletManager.getServiceInstance().registerActivity(pipeletHolder);
    } catch (final ProcessingException ex) {
      throw new CompilationException(VALIDATOR.createErrorCompilationMessage(key, "Error initialising pipelet: "
        + ex), ex);
    }
  }

  /**
   * create unique name for given key.
   */
  private static String getActivityKey(final OActivity activity) {
    return activity.getOwner().getName() + "/" + activity.name + "@" + activity.debugInfo.startLine;
  }

  /**
   * Extension activity to invoke simple pipelets.
   * 
   * @author jschumacher
   * 
   */
  public static class InvokePipeletActivity implements ExtensionOperation, ExtensionValidator {

    /**
     * promote invocation to pipelet manager.
     * 
     * @see org.apache.ode.bpel.extension.ExtensionOperation#run(java.lang.Object, org.w3c.dom.Element)
     */
    @Override
    public void run(final Object contexto, final Element element) throws FaultException {
      final ExtensionContext context = (ExtensionContext) contexto;
      invokeActivity(context, element);
    }

    /**
     * to be called during compilation of invokePipelet activities.
     * 
     * @see org.apache.ode.bpel.extension.ExtensionValidator#validate(java.lang.Object,
     *      org.apache.ode.bpel.extension.ExtensibleElement)
     */
    @Override
    public void validate(final Object compiler, final ExtensibleElement element) {
      final CompilerContext compilerContext = (CompilerContext) compiler;
      final OExtensionActivity activity = (OExtensionActivity) compilerContext.getCurrent();
      registerActivity(activity, element, compilerContext);
    }
  }

}
