/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/

package org.eclipse.smila.processing.pipelets.test;

import java.io.File;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.io.FileUtils;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.pipelets.FileReaderPipelet;
import org.eclipse.smila.processing.util.ProcessingConstants;

/**
 * Test class for FileReader pipelet.
 */
public class TestFileReaderPipelet extends APipeletTest {

  /** test content for the test file. */
  private static final String TEST_CONTENT = "blabla\nblub blub\n\u00e4\u00f6\n"
    + "Lorem ipsum dolor sit amet, consectetur adipisici elit, "
    + "sed eiusmod tempor incidunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, "
    + "quis nostrud exercitation ullamco laboris nisi ut aliquid ex ea commodi consequat. "
    + "Quis aute iure reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. "
    + "Excepteur sint obcaecat cupiditat non proident, sunt in culpa qui officia deserunt "
    + "mollit anim id est laborum.\nDuis autem vel eum iriure dolor in hendrerit in vulputate velit "
    + "esse molestie consequat, vel illum dolore eu feugiat nulla facilisis at vero eros et "
    + "accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue duis "
    + "dolore te feugait nulla facilisi. Lorem ipsum dolor sit amet, consectetuer adipiscing elit, "
    + "sed diam nonummy nibh euismod tincidunt ut laoreet dolore magna aliquam erat volutpat.\n"
    + "Ut wisi enim ad minim veniam, quis nostrud exerci tation ullamcorper suscipit lobortis nisl "
    + "ut aliquip ex ea commodo consequat. Duis autem vel eum iriure dolor in hendrerit in "
    + "vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla facilisis at vero "
    + "eros et accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue "
    + "duis dolore te feugait nulla facilisi.\n";

  /** The file reader pipelet. */
  private FileReaderPipelet _fileReaderPipelet;

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _fileReaderPipelet = new FileReaderPipelet();
    assertNotNull(_fileReaderPipelet);

    final AnyMap configuration = _blackboard.getDataFactory().createAnyMap();
    configuration.put(FileReaderPipelet.PROPERTY_PATH_ATTRIBUTE, "path");
    configuration.put(FileReaderPipelet.PROPERTY_CONTENT_ATTACHMENT, "content");
    configuration.put(ProcessingConstants.KEY_FAIL_ON_ERROR, true);
    _fileReaderPipelet.configure(configuration);
  }

  /**
   * Test reading when file does not exist.
   * 
   * @throws Exception
   *           if any error occurs
   */
  public void testReadingFileWithError() throws Exception {
    final String idValue = "readIdWithError";

    final Record record = _blackboard.getRecord(idValue, Get.AUTO_CREATE);
    final AnyMap metaData = record.getMetadata();
    metaData.put("path", "DontExistingPath");
    _blackboard.setRecord(record);

    final String[] recordIds = new String[] { record.getId() };
    try {
      _fileReaderPipelet.process(_blackboard, recordIds);
      fail("Must throw ProcessingException due to non existing file.");
    } catch (final ProcessingException e) {
      ;// nothing
    }
  }

  /**
   * Test reading with existing file.
   * 
   * @throws Exception
   *           if any error occurs
   */
  public void testReadingFile() throws Exception {
    final String idValue = "readId";
    final Record record = _blackboard.getRecord(idValue, Get.AUTO_CREATE);
    final AnyMap metaData = record.getMetadata();
    final File testFile = File.createTempFile(getName(), ".tmp");
    final FileOutputStream fos = new FileOutputStream(testFile);
    fos.write(TEST_CONTENT.getBytes("utf-8"));
    fos.flush();
    fos.close();
    metaData.put("path", testFile.getCanonicalPath());
    _blackboard.setRecord(record);
    final String[] recordIds = new String[] { record.getId() };
    final String[] result = _fileReaderPipelet.process(_blackboard, recordIds);
    assertNotNull(result);
    assertEquals(recordIds.length, result.length);
    final byte[] attachement = _blackboard.getAttachmentAsBytes(idValue, "content");
    assertNotNull(attachement);
    final String value = new String(attachement, "utf-8");
    assertTrue(value.length() > 1);
    assertEquals(TEST_CONTENT, value);
    FileUtils.deleteQuietly(testFile);
  }
  
  /** tests error handling in case of record failures. */
  public void testRobustness() throws Exception {    
    final FileReaderPipelet pipelet = new FileReaderPipelet();    
    final AnyMap configuration = _blackboard.getDataFactory().createAnyMap();
    // create configuration where required parameter PROPERTY_PATH_ATTRIBUTE is missing
    configuration.put(FileReaderPipelet.PROPERTY_CONTENT_ATTACHMENT, "content");
    configuration.put(ProcessingConstants.KEY_FAIL_ON_ERROR, true);
    pipelet.configure(configuration);
    
    final File testFile = File.createTempFile(getName(), ".tmp");
    final FileOutputStream fos = new FileOutputStream(testFile);
    fos.write(TEST_CONTENT.getBytes("utf-8"));
    fos.flush();
    fos.close();
                           
    final int noOfRecords = 9;    
    final Collection<String> recordIds = new ArrayList<String>();
    for (int i = 1; i <= noOfRecords; i++) {
      final String recordId = createBlackboardRecord("filter", Integer.toString(i));
      recordIds.add(recordId);
      if (i % 2 == 0) { // prepare correct records        
        _blackboard.getRecord(recordId).getMetadata().getMap("_parameters", true)
          .put(FileReaderPipelet.PROPERTY_PATH_ATTRIBUTE, "path");
        _blackboard.getRecord(recordId).getMetadata().put("path", testFile.getCanonicalPath());
      }
    }
    doRobustnessTest(pipelet, configuration, recordIds, noOfRecords, true);
    doRobustnessTest(pipelet, configuration, recordIds, noOfRecords, false);
  }
  
}
