/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.processing.test;

import java.util.Arrays;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/** Test for {@link ResultCollector} class. */
public class TestResultCollector extends DeclarativeServiceTestCase {

  /** exception used for failure testing. */
  private final Exception _testException = new RuntimeException();

  /** ParameterAccessor used for testing. */
  private ParameterAccessor _paramAccessor;

  /** Blackboard used for testing. */
  private Blackboard _blackboard;

  /** tests failOnError parameter in record. */
  public void testFailOnError() throws Exception {
    final ResultCollector collector = new ResultCollector(_paramAccessor, null, false);

    boolean failOnError = false;
    final String recordId = "id1";
    createBlackboardRecord(recordId, failOnError);
    collector.addFailedResult(recordId, _testException); // should not throw exception
    assertEquals(recordId, collector.getResultIds()[0]);

    failOnError = true;
    final String recordId2 = "id2";
    createBlackboardRecord(recordId2, failOnError);
    try {
      collector.addFailedResult(recordId2, _testException);
      fail();
    } catch (final ProcessingException e) {
      assertEquals(_testException, e.getCause());
    }
  }

  /** tests dropOnError setting in ResultCollector. */
  public void testDropOnError() throws Exception {
    final String failedId = "id_failed";
    createBlackboardRecord(failedId, false);
    createBlackboardRecord("id1", false);
    createBlackboardRecord("id2", false);

    boolean dropOnError = false;
    ResultCollector collector = new ResultCollector(_paramAccessor, null, dropOnError);
    collector.addResult("id1");
    collector.addFailedResult(failedId, _testException);
    collector.addResult("id2");
    assertEquals(3, collector.getResultIds().length);
    assertTrue("Result should contain id of failed record",
      Arrays.asList(collector.getResultIds()).contains(failedId));

    dropOnError = true;
    collector = new ResultCollector(_paramAccessor, null, dropOnError);
    collector.addResult("id1");
    collector.addFailedResult(failedId, _testException);
    collector.addResult("id2");
    assertEquals(2, collector.getResultIds().length);
    assertFalse("Result should not contain id of failed record",
      Arrays.asList(collector.getResultIds()).contains(failedId));
  }

  /**
   * create a new record on the blackboard with failOnError parameter.
   */
  private void createBlackboardRecord(final String recordId, final boolean failOnError)
    throws BlackboardAccessException {
    _blackboard.removeRecord(recordId);
    final Record record = _blackboard.getRecord(recordId, Get.NEW);
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(ProcessingConstants.KEY_FAIL_ON_ERROR, failOnError);
    record.getMetadata().put(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, parameters);
  }

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    final BlackboardFactory factory = getService(BlackboardFactory.class);
    _blackboard = factory.createTransientBlackboard();
    assertNotNull("no Blackboard created", _blackboard);
    _paramAccessor = new ParameterAccessor(_blackboard);
  }
}
