/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.processing.worker.test;

import java.io.ByteArrayInputStream;
import java.util.Collection;
import java.util.HashSet;

import org.eclipse.smila.binarystorage.BinaryStorageService;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.test.JobManagerTestBase;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.input.RecordInput;

public abstract class ProcessingWorkerTestBase extends JobManagerTestBase {
  protected static final String TEMP_STORE = "tempStore";

  protected static final String STORE = "store";

  protected static final String BUCKET_NAME = "outbulkBucket";

  protected BulkbuilderService _builder;

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _builder = getService(BulkbuilderService.class);
    assertNotNull(_builder);
    // wait for binary storage...
    assertNotNull(getService(BinaryStorageService.class));
    // create stores...
    _objectStoreService.ensureStore(STORE);
    _objectStoreService.ensureStore(TEMP_STORE);
    BpelWorkerTestPipelet.s_throwExceptions = true;
  }

  protected void executeJob(final String workflowName, final int noOfRecords) throws Exception {
    final String jobName = "testJob";
    createJob(workflowName, jobName);
    final String jobRunId = startJob(jobName);
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, DataFactory.DEFAULT.createRecord(Integer.toString(i)));
    }
    commitAndWaitForJob(jobName, jobRunId, true);
  }

  protected abstract void createJob(String workflowName, String jobName) throws Exception;

  protected void assertRecordsWithAttachmentsProcessed(final int expectedNoOfRecords, final String attachmentName,
    final boolean expectedHasAttachment, final boolean expectedAttachmentInMemory) throws Exception {
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(STORE, BUCKET_NAME);
    assertEquals(1, storeObjects.size());
    final BulkInfo bulkInfo = new BulkInfo(BUCKET_NAME, STORE, storeObjects.iterator().next().getId());
    final RecordInput recordInput = new RecordInput(bulkInfo, _objectStoreService);
    final Collection<String> ids = new HashSet<String>();
    try {
      Record record = recordInput.getRecord();
      while (record != null) {
        ids.add(record.getId());
        final String idAsString = String.valueOf(record.getId());
        final AnyMap metadata = record.getMetadata();
        assertEquals(idAsString, metadata.getStringValue(BpelWorkerAttachmentTestPipelet.ATTRIBUTE_ATTACHMENT_TEXT));
        assertEquals(expectedHasAttachment, record.hasAttachment(attachmentName));
        if (expectedHasAttachment) {
          assertEquals(idAsString, new String(record.getAttachmentAsBytes(attachmentName)));
        }
        final String attachmentStreamClass =
          metadata.getStringValue(BpelWorkerAttachmentTestPipelet.ATTRIBUTE_ATTACHMENT_STREAMCLASS);
        // If the attachment stream was a ByteArrayInputStream, the attachment was kept in memory.
        // Yes, that's probably not a nice way to check this, but it works, and I don't want to extend the
        // blackboard API to be able to check this.
        assertEquals(expectedAttachmentInMemory,
          attachmentStreamClass.equals(ByteArrayInputStream.class.getSimpleName()));
        record = recordInput.getRecord();
      }
    } finally {
      recordInput.close();
      _objectStoreService.removeObject(STORE, bulkInfo.getObjectName());
    }
    assertEquals(expectedNoOfRecords, ids.size());
  }

  protected void pushRecordsWithAttachment(final String jobName, final String jobRunId, final int noOfRecords,
    final String attachmentName) throws Exception {
    for (int i = 0; i < noOfRecords; i++) {
      final Record record = DataFactory.DEFAULT.createRecord(Integer.toString(i));
      record.getMetadata().put(BpelWorkerAttachmentTestPipelet.ATTRIBUTE_ATTACHMENT_NAME, attachmentName);
      record.setAttachment(attachmentName, Integer.toString(i).getBytes("UTF-8"));
      _builder.addRecord(jobName, record);
    }
    commitAndWaitForJob(jobName, jobRunId, true);
  }

  /**
   * @param jobName
   * @param jobRunId
   * @param expectJobSuccessful
   * @throws BulkbuilderException
   * @throws JobManagerException
   * @throws Exception
   */
  protected void commitAndWaitForJob(final String jobName, final String jobRunId, final boolean expectJobSuccessful)
    throws BulkbuilderException, JobManagerException, Exception {
    _builder.commitJob(jobName);
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId, 100000); // wait max. 10 seconds
    Thread.sleep(1000); // give jobmanager some time to do final clean up of old job run 
    if (expectJobSuccessful) {
      assertJobRunSucceeded(jobName, jobRunId, 1);
    } else {
      assertJobRunFailed(jobName, jobRunId);
    }
  }

}
