/**
 * 
 */
package org.eclipse.smila.solr;

import junit.framework.TestCase;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.common.params.FacetParams;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter.FacetDateOther;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter.FacetMethod;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter.FacetSort;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter.FacetType;

/**
 * Facet test class.
 * 
 * @author pwissel
 * 
 */
public class Facet_Test extends TestCase {

  /**
   * Dummy workflow.
   */
  private final static String _workflow = "testWorkflow";

  /**
   * Dummy attribute.
   */
  private final static String _attribute = "testAttribute";

  /**
   * The log.
   */
  private final Log _log = LogFactory.getLog(Facet_Test.class);

  /**
   * Test FacetRecordConfig_Exception.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_Exception() throws Exception {
    try {
      new FacetQueryConfigAdapter(null);
      fail("Should throw NotImplementedException");
    } catch (Exception exception) {
      ;
    }
  }

  /**
   * Test FacetRecordConfig_Solr.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_Solr() throws Exception {
    final FacetQueryConfigAdapter config = new FacetQueryConfigAdapter(FacetType.GLOBAL);
    final QueryBuilder builder = new QueryBuilder(_workflow);
    builder.addFacetByConfig(SolrConstants.GLOBAL, config.getAnyMap());

    final Record record = builder.getQuery();
    final AnyMap metadata = record.getMetadata();
    final AnyMap facetby = metadata.getSeq(QueryConstants.FACETBY).getMap(0);

    assertEquals(SolrConstants.GLOBAL, facetby.getStringValue(QueryConstants.ATTRIBUTE));
    assertTrue(facetby.getBooleanValue(FacetParams.FACET));
    assertEquals(2, facetby.size());

    if (_log.isDebugEnabled()) {
      _log.debug(XmlSerializationUtils.serialize2string(record));
    }
  }

  /**
   * Test FacetRecordConfig_SolrParameter.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_SolrParameter() throws Exception {
    final FacetQueryConfigAdapter config = new FacetQueryConfigAdapter(FacetType.GLOBAL);
    config.setFacetEnumCacheMinDf(1);
    config.setFacetLimit(10);
    config.setFacetMethod(FacetMethod.ENUM);
    config.setFacetMinCount(1);
    config.setFacetMissing(true);
    config.setFacetOffset(0);
    config.setFacetPrefix("pre");
    config.setFacetSort(FacetSort.COUNT);
    final QueryBuilder builder = new QueryBuilder(_workflow);
    builder.addFacetByConfig(SolrConstants.GLOBAL, config.getAnyMap());

    final Record record = builder.getQuery();
    final AnyMap metadata = record.getMetadata();
    final AnyMap facetby = metadata.getSeq(QueryConstants.FACETBY).getMap(0);

    assertEquals(1, facetby.getLongValue(FacetParams.FACET_ENUM_CACHE_MINDF).intValue());
    assertEquals(10, facetby.getLongValue(FacetParams.FACET_LIMIT).intValue());
    assertEquals(FacetMethod.ENUM.toString(), facetby.getStringValue(FacetParams.FACET_METHOD));
    assertEquals(1, facetby.getLongValue(FacetParams.FACET_MINCOUNT).intValue());
    assertTrue(facetby.getBooleanValue(FacetParams.FACET_MISSING));
    assertEquals(0, facetby.getLongValue(FacetParams.FACET_OFFSET).intValue());
    assertEquals("pre", facetby.getStringValue(FacetParams.FACET_PREFIX));
    assertEquals(FacetSort.COUNT.toString(), facetby.getStringValue(FacetParams.FACET_SORT));
    assertEquals(10, facetby.size());

    if (_log.isDebugEnabled()) {
      _log.debug(XmlSerializationUtils.serialize2string(record));
    }
  }

  /**
   * Test FacetRecordConfig_Field.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_Field() throws Exception {
    final FacetQueryConfigAdapter config = new FacetQueryConfigAdapter(FacetType.FIELD);
    final QueryBuilder builder = new QueryBuilder(_workflow);
    builder.addFacetByConfig(_attribute, config.getAnyMap());

    final Record record = builder.getQuery();
    final AnyMap metadata = record.getMetadata();
    final AnyMap facetby = metadata.getSeq(QueryConstants.FACETBY).getMap(0);

    assertEquals(_attribute, facetby.getStringValue(QueryConstants.ATTRIBUTE));
    assertEquals(FacetParams.FACET_FIELD, facetby.getStringValue(SolrConstants.FACET_TYPE));
    assertEquals(1, metadata.getSeq(QueryConstants.FACETBY).size());

    if (_log.isDebugEnabled()) {
      _log.debug(XmlSerializationUtils.serialize2string(record));
    }
  }

  /**
   * Test FacetRecordConfig_Query.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_Query() throws Exception {
    String[] query = { "* TO 1000", "1000 TO 2000", "2000 TO *" };
    final FacetQueryConfigAdapter config = new FacetQueryConfigAdapter(FacetType.QUERY, query);
    final QueryBuilder builder = new QueryBuilder(_workflow);
    builder.addFacetByConfig(_attribute, config.getAnyMap());

    final Record record = builder.getQuery();
    final AnyMap metadata = record.getMetadata();
    final AnyMap facetby = metadata.getSeq(QueryConstants.FACETBY).getMap(0);

    assertEquals(FacetParams.FACET_QUERY, facetby.getStringValue(SolrConstants.FACET_TYPE));
    assertNotNull(facetby.getSeq(SolrConstants.FACET_QUERY_ATTR));
    assertEquals(3, facetby.getSeq(SolrConstants.FACET_QUERY_ATTR).size());
    final String expected = StringUtils.join(query);
    final String actual = StringUtils.join(facetby.getSeq(SolrConstants.FACET_QUERY_ATTR).toArray());
    assertEquals(expected, actual);

    if (_log.isDebugEnabled()) {
      _log.debug(XmlSerializationUtils.serialize2string(record));
    }
  }

  /**
   * Test FacetRecordConfig_QueryException.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_QueryTypeReqiresQuery() throws Exception {
    try {
      new FacetQueryConfigAdapter(FacetType.QUERY);
      fail("expected IllegalArgumentException");
    } catch (IllegalArgumentException exception) {
      ;
    }

    try {
      new FacetQueryConfigAdapter(FacetType.QUERY, null);
      fail("Should throw IllegalArgumentException");
    } catch (IllegalArgumentException exception) {
      ;
    }

    try {
      new FacetQueryConfigAdapter(FacetType.QUERY, (String[]) null);
      fail("Should throw IllegalArgumentException");
    } catch (IllegalArgumentException exception) {
      ;
    }
  }

  /**
   * Test FacetRecordConfig_Date.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_Date() throws Exception {
    final FacetQueryConfigAdapter config = new FacetQueryConfigAdapter(FacetType.DATE);
    config.setFacetDateEnd("31.12.2010");
    config.setFacetDateGap("+1MONTH");
    config.setFacetDateHardend(true);
    config.setFacetDateOther(FacetDateOther.AFTER);
    config.setFacetDateStart("01.01.2010");
    final QueryBuilder builder = new QueryBuilder(_workflow);
    builder.addFacetByConfig(_attribute, config.getAnyMap());

    final Record record = builder.getQuery();
    final AnyMap metadata = record.getMetadata();
    final AnyMap facetby = metadata.getSeq(QueryConstants.FACETBY).getMap(0);

    assertEquals(FacetParams.FACET_DATE, facetby.getStringValue(SolrConstants.FACET_TYPE));
    assertEquals("31.12.2010", facetby.getStringValue(FacetParams.FACET_DATE_END));
    assertEquals("+1MONTH", facetby.getStringValue(FacetParams.FACET_DATE_GAP));
    assertTrue(facetby.getBooleanValue(FacetParams.FACET_DATE_HARD_END));
    assertEquals(FacetDateOther.AFTER.toString(), facetby.getStringValue(FacetParams.FACET_DATE_OTHER));
    assertEquals("01.01.2010", facetby.getStringValue(FacetParams.FACET_DATE_START));

    if (_log.isDebugEnabled()) {
      _log.debug(XmlSerializationUtils.serialize2string(record));
    }
  }

  /**
   * Test FacetRecordConfig_Multi.
   * 
   * @throws Exception
   *           Exception.
   */
  public void test_FacetRecordConfig_Multi() throws Exception {
    final FacetQueryConfigAdapter config1 = new FacetQueryConfigAdapter(FacetType.FIELD);
    final FacetQueryConfigAdapter config2 = new FacetQueryConfigAdapter(FacetType.DATE);
    final QueryBuilder builder = new QueryBuilder(_workflow);
    builder.addFacetByConfig(_attribute + "1", config1.getAnyMap());
    builder.addFacetByConfig(_attribute + "2", config2.getAnyMap());

    final Record record = builder.getQuery();
    final AnyMap metadata = record.getMetadata();
    final AnySeq facetby = metadata.getSeq(QueryConstants.FACETBY);

    assertEquals(2, facetby.size());

    if (_log.isDebugEnabled()) {
      _log.debug(XmlSerializationUtils.serialize2string(record));
    }
  }

}
