/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.solr.documentation;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.solr.SolrConstants;
import org.eclipse.smila.solr.search.SolrQueryBuilder;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter.FacetType;
import org.eclipse.smila.solr.util.HighlightingQueryConfigAdapter;
import org.eclipse.smila.solr.util.SpellCheckQueryConfigAdapter;
import org.eclipse.smila.solr.util.TermsQueryConfigAdapter;

/**
 * Example how to build a full featured Solr search record. See: http://wiki.eclipse.org/SMILA/Documentation/Solr.
 * 
 * @author pwissel
 * 
 */
public class SolrQueryBuilderExample {

  public static void main(String[] args) {
    // create solr specific query builder
    final SolrQueryBuilder builder = new SolrQueryBuilder();

    // set query
    builder.setQuery("query");

    // set start (equals offset, default: 0)
    builder.setStart(10);

    // set rows (equals max count, default: 10)
    builder.setRows(5);

    // set fields (equals result attributes, default: Id, score)
    final String[] fl = { "Path", "Size", "Content" };
    builder.addFields(fl);

    // add a filter query (example: size between 500 and 1000)
    builder.addFilterQuery("Size:[500 TO 1000]");

    // set shards
    final String[] shards = { "http://localhost:8983/solr", "http://remote-server:8983/solr" };
    builder.setShards(shards);

    // set request handler
    builder.setRequestHandler("/terms");

    // create global highlighting configuration (required, enables highlighting)
    final HighlightingQueryConfigAdapter highlighting = new HighlightingQueryConfigAdapter();
    highlighting.setHighlightingFields("Content");
    highlighting.setHighlightingSimplePre("<b>");
    highlighting.setHighlightingSimplePost("</b>");
    builder.addHighlightingConfiguration(highlighting);

    // create global facet configuration (required, enables facets)
    final FacetQueryConfigAdapter facet_global = new FacetQueryConfigAdapter(FacetType.GLOBAL);
    builder.addFacetConfiguration(SolrConstants.GLOBAL, facet_global);

    // create field facet configuration
    final FacetQueryConfigAdapter facet_field = new FacetQueryConfigAdapter(FacetType.FIELD);
    facet_field.setFacetLimit(10);
    builder.addFacetConfiguration("Extension", facet_field);

    // create facet date configuration
    final FacetQueryConfigAdapter facet_date = new FacetQueryConfigAdapter(FacetType.DATE);
    facet_date.setFacetDateStart("NOW/DAY-5DAYS");
    facet_date.setFacetDateGap("+1DAY");
    facet_date.setFacetDateEnd("NOW/DAY+1DAY");
    builder.addFacetConfiguration("LastModifiedDate", facet_date);

    // create facet query configuration (range example)
    final String[] fq = { "* TO 1000", "1000 TO 5000", "5000 * TO" };
    final FacetQueryConfigAdapter facet_query = new FacetQueryConfigAdapter(FacetType.QUERY, fq);
    builder.addFacetConfiguration("Size", facet_query);

    // create terms configuration (auto suggest example)
    final TermsQueryConfigAdapter terms = new TermsQueryConfigAdapter("Title");
    terms.setTermsLower("auto");
    terms.setTermsPrefix("sug");
    builder.setTermsConfiguration(terms);

    // create spell check configuration
    final SpellCheckQueryConfigAdapter spellcheck = new SpellCheckQueryConfigAdapter();
    spellcheck.setSpellCheckCount(5);
    spellcheck.setSpellCheckExtendedResults(true);
    spellcheck.setSpellCheckCollate(true);
    builder.setSpellCheckConfiguration(spellcheck);

    // create xml output
    final Record record = builder.getQuery();
    final String xml = XmlSerializationUtils.serialize2string(record);
    System.out.println(xml);
  }

}
