/*******************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Thomas Menzel (brox IT Solution GmbH) - initial creator
 *******************************************************************************/
package org.eclipse.smila.solr.manual;

import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.impl.BinaryRequestWriter;
import org.apache.solr.client.solrj.impl.CommonsHttpSolrServer;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.FacetField.Count;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.params.FacetParams;

/**
 * these tests investigate how things are done with Solr(J).
 * 
 * They operate on a freshly unzipped install of Solr 1.4.1 (and possibly later too) where the example data has been
 * added like so: <tt>example\exampledocs$ java -jar post.jar *.xml</tt>
 * 
 * @see http://wiki.apache.org/solr/Solrj
 * @see if ref' the book "Book Solr 1.4, Enterprise Search Server" like so: [p. {page}]
 * @author tmenzel
 */
public class Solr_InvestigationTest extends TestCase {

  /**
   * @author tmenzel
   * 
   */
  private final class FindCountPredicate implements Predicate {
    private final String _name;

    /**
     * @param string
     */
    public FindCountPredicate(String name) {
      _name = name;
      // TODO Auto-generated constructor stub
    }

    @Override
    public boolean evaluate(Object arg0) {
      Count count = (Count) arg0;
      return count.getName().equals(_name);

    }
  }

  /** The logger. */
  private final org.apache.commons.logging.Log _log = org.apache.commons.logging.LogFactory.getLog(this.getClass());

  private CommonsHttpSolrServer _server;

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    _server = new CommonsHttpSolrServer("http://localhost:8983/solr");
    _server.setRequestWriter(new BinaryRequestWriter());

  }

  /**
   * tests that facet field specific config (.limit) has no side effect on another field.
   * 
   * the facet params can be set globally (ie for all) or per field, which is recommended [p. 147]
   */
  public void test_Search_Facetting_FacetFieldConfig() throws Exception {
    SolrQuery query = new SolrQuery().//
      setQuery("*:*").//
      addField("name").//
      addFacetField("cat").//
      addFacetField("payloads").//
      setFacet(true);

    /*
     * this is how a config is applied to just one facet field, and not all. in this case we limit the amount of facet
     * values returned by the searach. so far i havent found a nice API to set those contraints per field other than
     * this.
     */
    final int LIMITED_FACET_COUNT = 1;
    query.add("f.payloads." + FacetParams.FACET_LIMIT, LIMITED_FACET_COUNT + "");

    final QueryResponse response = _server.query(query);

    assertEquals("numFound doesnt match!. did u index the rigth stuff?", 19, response.getResults().getNumFound());

    // verify that the limit only applies to payloads and not cat
    assertEquals(16, response.getFacetField("cat").getValueCount()); //
    assertEquals(LIMITED_FACET_COUNT, response.getFacetField("payloads").getValueCount()); //

  }

  /*
   * TODO | TM | test - how to filter query based on a selected face
   * 
   * see: Count.getAsFilterQuery
   * 
   * | TM @ Feb 15, 2011
   */

  /**
   * this mimics http://localhost:8983/solr/select/?indent=on&q=*:*&fl=name&facet=true&facet.field=cat
   */
  public void test_Search_Facetting_Text_Simple() throws Exception {

    SolrQuery query = new SolrQuery().//
      setQuery("*:*").//
      addField("name").//
      addFacetField("cat").//
      setFacet(true);

    final QueryResponse response = _server.query(query);

    assertEquals("numFound doesnt match!. did u index the rigth stuff?", 19, response.getResults().getNumFound());

    // cat facets
    final FacetField facetField = response.getFacetField("cat");

    assertEquals(16, facetField.getValueCount()); //
    final Count electronicsCount = getFacetCount(facetField, "electronics");
    assertNotNull(electronicsCount);
    assertEquals(17, electronicsCount.getCount());
  }

  /**
   * tests what is returned as a filter for a facet value.
   */
  public void test_Search_Filter_FromFacetting() throws Exception {
    SolrQuery query = new SolrQuery().//
      setQuery("*:*").//
      addField("name").//
      addFacetField("cat").//
      setFacet(true);

    final QueryResponse response = _server.query(query);
    assertEquals("numFound doesnt match!. did u index the rigth stuff?", 19, response.getResults().getNumFound());

    final FacetField facetField = response.getFacetField("cat");
    final Count electronicsCount = getFacetCount(facetField, "electronics");
    final String asFilterQuery = electronicsCount.getAsFilterQuery();

    _log.debug(asFilterQuery);

    assertEquals("cat:electronics", asFilterQuery);
  }

  /**
   * @param facetField
   * @param facetName
   *          TODO
   * @return
   */
  private Count getFacetCount(final FacetField facetField, String facetName) {
    final List<Count> facets = facetField.getValues();
    final Count electronicsCount = (Count) CollectionUtils.find(facets, new FindCountPredicate(facetName));
    return electronicsCount;
  }

  /**
   * tests how faceting over numbers work.
   * <p/>
   * this tests facets over 2 diff number fields, one is an int the other a float.
   * <p/>
   * conclusion: with facet query one cat fetch arbitray facet values in given ranges. the result is rather generic and
   * would need to be parsed into an object to return the field name, range value and counts.
   * 
   * @see [p. 152]
   */
  public void test_Search_Facetting_TwoNumberFields() throws Exception {
    SolrQuery query = new SolrQuery().//
      setQuery("*:*").//
      setRows(0).//

      /*
       * PERF consideration: when one has >1 num fields to facet for the same query it is probably faster to get them
       * all in one go as opposed to send multiple queries. rationale: the base solr query over which the facets are
       * determined needs only be executed once. | TM @ Feb 15, 2011
       */
      addFacetQuery("popularity:[* TO 3]").//
      addFacetQuery("popularity:[4 TO 7]").//
      addFacetQuery("popularity:[8 TO *]").//

      addFacetQuery("weight:[* TO 10]").//
      addFacetQuery("weight:[11 TO 100]").//
      addFacetQuery("weight:[101 TO *]").//
      setFacet(true);

    final QueryResponse response = _server.query(query);
    assertEquals("numFound doesnt match!. did u index the rigth stuff?", 19, response.getResults().getNumFound());

    final Map<String, Integer> facetQueryFacets = response.getFacetQuery();

    _log.debug(ToStringBuilder.reflectionToString(facetQueryFacets));

    assertFacetQueryFacet(facetQueryFacets, 3, "popularity:[* TO 3]");
    assertFacetQueryFacet(facetQueryFacets, 4, "weight:[* TO 10]");

  }

  /**
   * shows that facetQuery can be used for an arbitrary faceting.
   * <p/>
   * possible use case: hierarchical facets which are in 2 index fields
   * 
   * @see [p. 152]
   */
  public void test_Search_Facetting_Arbitray() throws Exception {
    SolrQuery query = new SolrQuery().//
      setQuery("*:*").//
      setRows(0).//
      addFacetQuery("popularity:[* TO 3] AND " + "weight:[* TO 10]").//
      setFacet(true);

    final QueryResponse response = _server.query(query);
    assertEquals("numFound doesnt match!. did u index the rigth stuff?", 19, response.getResults().getNumFound());

    final Map<String, Integer> facetQueryFacets = response.getFacetQuery();

    _log.debug(ToStringBuilder.reflectionToString(facetQueryFacets));

    assertFacetQueryFacet(facetQueryFacets, 2, "popularity:[* TO 3] AND " + "weight:[* TO 10]");
  }

  /**
   * 
   */
  private void assertFacetQueryFacet(final Map<String, Integer> facetQuery, int expectedOccurence, String requestedRange) {
    final Integer count = facetQuery.get(requestedRange);
    assertNotNull("facet doesnt exit: " + requestedRange, count);
    assertEquals(expectedOccurence, count.intValue());
  }

  /* TODO | TM | filtering from facet values, single and multi select, exluding the selected values | TM @ Feb 15, 2011 */
  /*
   * TODO | TM | be able to dynamically adjust the facet value bounds for numeric fields. hint: [p. 189 stats] | TM @
   * Feb 15, 2011
   */

  /**
   * tests that a field containg >1 values (multi value field) returns both facets
   */
  public void test_Search_Facet_OnMultiValueField() throws Exception {
    SolrQuery query = new SolrQuery().//
      setQuery("id:SP2514N").// see hd.xml
      addFacetField("cat").//
      setFacet(true);

    final QueryResponse response = _server.query(query);

    assertEquals("numFound doesnt match!. did u index the rigth stuff?", 1, response.getResults().getNumFound());

    final FacetField facetField = response.getFacetField("cat");
    assertFacetFieldCount(facetField, "electronics", 1);
    /*
     * NOTE: note: although the hard drive is the item and one would expect that the facet value is "hard drive" this is
     * not so due to the tokenizing happening on that field. in real life one would not tokenize on this field or at
     * least have another that returns the string like so for display. since i work with the sample data here it just
     * work wiht it the way it is. | tmenzel @ Feb 24, 2011
     */
    assertFacetFieldCount(facetField, "hard", 1);
    assertFacetFieldCount(facetField, "drive", 1);

  }

  /**
   * asserts the count for a (normal) facet value
   */
  private void assertFacetFieldCount(final FacetField facetField, String facetValue, int expectedCount) {
    final Count facetCount = getFacetCount(facetField, facetValue);
    assertEquals("facet count doesnt match: " + facetField.getName(), expectedCount, facetCount.getCount());
  }
}
