/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.search;

import static org.apache.commons.lang.StringUtils.isBlank;

import java.util.UUID;

import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.MoreLikeThisParams;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.search.api.SearchService;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.solr.SolrConstants;
import org.eclipse.smila.solr.util.FacetQueryConfigAdapter;
import org.eclipse.smila.solr.util.HighlightingQueryConfigAdapter;
import org.eclipse.smila.solr.util.SpellCheckQueryConfigAdapter;
import org.eclipse.smila.solr.util.TermsQueryConfigAdapter;

/**
 * This class is used to build a Solr Search Record via API and execute the search/request.
 * 
 * @author pwissel
 * 
 */
public class SolrQueryBuilder extends QueryBuilder {

  /**
   * The default workflow.
   */
  private static final String DEFAULT_WORKFLOW = "SolrSearchDefaultWorkflow";

  /**
   * The default record id.
   */
  private static final String DEFAULT_RECORD_ID = "SolrSearchRecordId: ";

  /**
   * Default constructor.
   */
  public SolrQueryBuilder() {
    super(DEFAULT_WORKFLOW);
    final String id = DEFAULT_RECORD_ID + UUID.randomUUID();
    setId(id);
  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow workflow name.
   */
  public SolrQueryBuilder(String workflowName) {
    super(workflowName);

  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow name.
   * @param factory
   *          the data factory.
   */
  public SolrQueryBuilder(String workflowName, DataFactory factory) {
    super(workflowName, factory);
  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow name.
   * @param request
   *          the request record.
   */
  public SolrQueryBuilder(String workflowName, Record request) {
    super(workflowName, request);
  }

  /**
   * Set start parameter.
   * 
   * @param start
   *          the start parameter.
   * @return the solr query builder.
   */
  public SolrQueryBuilder setStart(int start) {
    setOffset(start);
    return this;
  }

  /**
   * Set rows parameter.
   * 
   * @param rows
   *          the rows parameter.
   * @return the solr query builder.
   */
  public SolrQueryBuilder setRows(int rows) {
    setMaxCount(rows);
    return this;
  }

  /**
   * Add fields parameter.
   * 
   * @param fields
   *          the fields parameter.
   * @return the solr query record.
   */
  public SolrQueryBuilder addFields(String... fields) {
    addResultAttributes(fields);
    return this;
  }

  /**
   * Set the request handler.
   * 
   * @param name
   *          The value given should start with a leading /. However, if it is missing it will be added. Blank values
   *          will result in removing the request handler parameter (reset to default).
   * @return the solr query builder.
   */
  public SolrQueryBuilder setRequestHandler(String name) {
    if (isBlank(name)) {
      getSolrQueryMap().remove(CommonParams.QT);
    } else {
      getSolrQueryMap().put(CommonParams.QT, name);
    }
    return this;
  }

  /**
   * Add a highlighting configuration.
   * 
   * @param config
   *          the highlighting configuration.
   * @return the solr query record,
   */
  public SolrQueryBuilder addHighlightingConfiguration(HighlightingQueryConfigAdapter config) {
    getSolrQueryMap().getSeq(SolrConstants.HIGHLIGHTING, true).add(config.getAnyMap());
    return this;
  }

  /**
   * Adds a native solr filter query.
   * 
   * @param filterQuery
   *          the filter query.
   * @return the solr query builder.
   */
  public SolrQueryBuilder addFilterQuery(String filterQuery) {
    if (filterQuery != null) {
      getSolrQueryMap().getSeq(SolrConstants.FILTER_QUERY, true).add(filterQuery);
    }
    return this;
  }

  /**
   * Add a facet configuration for the given core field or set global/default parameters with
   * {@link SolrConstants#GLOBAL}. Note that u always must do one call with at least teh following to enable faceting at
   * all.
   * 
   * <pre>
   * addFacetConfiguration(SolrConstants.FACET_GLOBAL_ATTR, new FacetQueryConfigAdapter(FacetType.SOLR))
   * </pre>
   * 
   * @param coreField
   *          the attribute name.
   * @param config
   *          the facet configuration.
   * @return the solr query builder.
   * @deprecated as of 1.0 use the base class, the format is not supported anymore by the search
   */
  @Deprecated
  public SolrQueryBuilder addFacetConfiguration(String coreField, FacetQueryConfigAdapter config) {
    addFacetByConfig(coreField, config.getAnyMap());
    return this;
  }

  /**
   * Set the terms configuration.
   * 
   * @param config
   *          the terms configuration.
   * @return the solr query builder.
   */
  public SolrQueryBuilder setTermsConfiguration(TermsQueryConfigAdapter config) {
    getSolrQueryMap().getMap(SolrConstants.TERMS, true).putAll(config.getAnyMap());
    return this;
  }

  /**
   * Get the solr query map _solr.query.
   * 
   * @return the solr query map.
   */
  AnyMap getSolrQueryMap() {
    return getMetadata().getMap(SolrConstants.QUERY_MAP, true);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.search.api.helper.QueryBuilder#executeRequest(org.eclipse.smila.search.api.SearchService)
   */
  @Override
  public SolrResultAccessor executeRequest(SearchService searchService) throws ProcessingException {
    super.executeRequest(searchService);
    return new SolrResultAccessor(getWorkflowName(), getQuery());
  }

  /**
   * Set the *:* as the query string to find all documents. Mainly used for testing. Only works when used with the right
   * (default) query handler.
   * 
   * @return SolrQueryBuilder.
   */
  public SolrQueryBuilder setQueryFindAll() {
    setQuery("*:*");
    return this;
  }

  /**
   * Set shards.
   * 
   * @param shards
   *          the shards.
   * @return SolrQueryBuilder.
   */
  public SolrQueryBuilder setShards(String... shards) {
    for (String shard : shards) {
      getSolrQueryMap().getSeq(SolrConstants.SHARDS, true).add(shard);
    }
    return this;
  }

  /**
   * Set the spellcheck configuration.
   * 
   * @param config
   *          the spellcheck configuraiton.
   * @return SolrQueryBuilder.
   */
  public SolrQueryBuilder setSpellCheckConfiguration(SpellCheckQueryConfigAdapter config) {
    getSolrQueryMap().getMap(SolrConstants.SPELLCHECK, true).putAll(config.getAnyMap());
    return this;
  }

  /**
   * This will add the parameters to the _solr.query map and name it {@link SolrConstants#MORE_LIKE_THIS}. If the map
   * dosnt contain a value {@code mlt} then it will be added setting it to true, thus turning on MLT.
   * 
   * @param mltArgs
   */
  public void setMoreLikeThis(AnyMap mltArgs) {
    getSolrQueryMap().put(SolrConstants.MORE_LIKE_THIS, mltArgs);
    final Value value = mltArgs.getValue(MoreLikeThisParams.MLT);
    if (value == null) {
      mltArgs.put(MoreLikeThisParams.MLT, true);
    }

  }

}
