/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.persistence.zk.test;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.zookeeper.ZkConnection;
import org.eclipse.smila.zookeeper.ZooKeeperService;

import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException.Cause;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskList;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.taskmanager.persistence.zk.TaskStorageZk;
import org.eclipse.smila.taskmanager.persistence.zk.ZkTaskQueue;

/**
 * test basic functionality of ZkTaskQueue. 
 */
public class TestZkTaskQueue extends DeclarativeServiceTestCase {
  /** ZooKeeper service. */
  private ZooKeeperService _zk;

  /**
   * test if ZooKeeper service was successfully started and registered.
   * 
   * @throws Exception
   *           no service found.
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _zk = getService(ZooKeeperService.class);
    assertNotNull(_zk);
  }

  /**
   * @throws Exception
   *           test fails
   */
  public void testBasic() throws Exception {
    final String queueName = "testBasic";
    final ZkTaskQueue queue = createQueue(queueName);
    final Task putTask = createTask(1, queueName);
    queue.put(putTask);
    assertNull(queue.get("partition", null));
    final Task getTask = queue.get(null, null);
    assertNotNull(getTask);
    assertEqualTasks(putTask, getTask);
    final Task ipTask = queue.getInProgressTask(getTask.getTaskId());
    assertNotNull(ipTask);
    assertEqualTasks(putTask, ipTask);
    queue.delete(getTask.getTaskId());
    assertNull(queue.get(null, null));
  }

  /**
   * @throws Exception
   *           test fails
   */
  public void testPutInProgress() throws Exception {
    final String queueName = "testPutInProgress";
    final ZkTaskQueue queue = createQueue(queueName);
    final Task putTask = createTask(1, queueName);
    queue.putInProgress(putTask);
    assertNull(queue.get("partition", null));
    assertNull(queue.get(null, null));
    final Task ipTask = queue.getInProgressTask(putTask.getTaskId());
    assertNotNull(ipTask);
    assertEqualTasks(putTask, ipTask);
    queue.delete(ipTask.getTaskId());
    assertNull(queue.get(null, null));
  }

  /**
   * @throws Exception
   *           test fails
   */
  public void testCounter() throws Exception {
    final String queueName = "testCounter";
    final ZkTaskQueue queue = createQueue(queueName);
    final Task putTask = createTask(1, queueName);
    queue.put(putTask);
    TaskCounter counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 1, 0);
    final Task getTask = queue.get(null, null);
    assertNotNull(getTask);
    assertEqualTasks(putTask, getTask);
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 1);
    queue.getInProgressTask(getTask.getTaskId());
    queue.delete(getTask.getTaskId());
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 0);
    assertNull(queue.get(null, null));
  }

  /**
   * @throws Exception
   *           test fails
   */
  public void testKeepAlive() throws Exception {
    final String queueName = "testKeepAlive";
    final ZkTaskQueue queue = createQueue(queueName);
    final Task putTask = createTask(1, queueName);
    queue.put(putTask);
    assertNull(queue.get("partition", null));
    final Task getTask = queue.get(null, null);
    assertNotNull(getTask);
    assertEqualTasks(putTask, getTask);
    queue.keepAlive(getTask.getTaskId());
    queue.getInProgressTask(getTask.getTaskId());
    queue.delete(getTask.getTaskId());
    assertNull(queue.get(null, null));
    try {
      queue.keepAlive(getTask.getTaskId());
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception", ex instanceof TaskmanagerException);
    }
  }

  /**
   * check that task finishing can be retried in a new session.
   * 
   * @throws Exception
   */
  public void testRetryTaskFinishing() throws Exception {
    final String queueName = "testDeleteWithoutFinishingError";
    final ZkTaskQueue queue = createQueue(queueName);
    final Task putTask = createTask(1, queueName);
    queue.put(putTask);
    final Task getTask = queue.get(null, null);
    assertNotNull(getTask);
    assertEqualTasks(putTask, getTask);
    assertNotNull(queue.getInProgressTask(putTask.getTaskId()));
    queue.disconnectZkSession();
    assertNull(queue.get(null, null));
    assertNotNull(queue.getInProgressTask(putTask.getTaskId()));
    queue.delete(putTask.getTaskId());
    assertNull(queue.get(null, null));
  }

  /**
   * @throws Exception
   *           test fails
   */
  public void testTaskOrdering() throws Exception {
    final String queueName = "testTaskOrdering";
    final ZkTaskQueue queue = createQueue(queueName);
    final int noOfTasks = 16;
    final Task[] putTasks = new Task[noOfTasks];
    for (int i = 0; i < noOfTasks; ++i) {
      putTasks[i] = createTask(i, queueName);
      queue.put(putTasks[i]);
      final TaskCounter counter = queue.getTaskCounter();
      assertCounter(counter, queueName, i + 1, 0);
    }
    for (int i = 0; i < noOfTasks; ++i) {
      assertNull(queue.get("partition", null));
      final Task getTask = queue.get(null, null);
      assertNotNull(getTask);
      assertEqualTasks(putTasks[i], getTask);
      TaskCounter counter = queue.getTaskCounter();
      assertCounter(counter, queueName, noOfTasks - (i + 1), 1);
      queue.getInProgressTask(getTask.getTaskId());
      queue.delete(getTask.getTaskId());
      counter = queue.getTaskCounter();
      assertCounter(counter, queueName, noOfTasks - (i + 1), 0);
    }
    assertNull(queue.get(null, null));
  }

  /**
   * @throws Exception
   *           test fails
   */
  public void testQualifiedQueue() throws Exception {
    final String queueName = "testQualifiedQueue";
    final ZkTaskQueue queue = createQueue(queueName);
    assertFalse(queue.hasQualifiedTasks());
    final int noOfPartitions = 4;
    final int noOfTasksPerPartition = 4;
    final int noOfTasks = noOfPartitions * noOfTasksPerPartition;
    final Task[] putTasks = new Task[noOfTasks];
    for (int i = 0; i < noOfTasks; ++i) {
      putTasks[i] = createTask(i, queueName);
      final int partition = i % noOfPartitions;
      putTasks[i].setQualifier("parts/" + Integer.toString(partition));
      queue.put(putTasks[i]);
      final TaskCounter counter = queue.getTaskCounter();
      assertCounter(counter, queueName, i + 1, 0);
    }
    assertTrue(queue.hasQualifiedTasks());
    for (int i = 0; i < noOfTasks; ++i) {
      assertNull(queue.get(null, null));
      assertNull(queue.get("parts/partition", null));
      final int partition = i % noOfPartitions;
      final Task getTask = queue.get("parts/" + Integer.toString(partition), null);
      assertNotNull(getTask);
      assertEqualTasks(putTasks[i], getTask);
      final Task ipTask = queue.getInProgressTask(getTask.getTaskId());
      assertNotNull(ipTask);
      assertEqualTasks(putTasks[i], ipTask);
      TaskCounter counter = queue.getTaskCounter();
      assertCounter(counter, queueName, noOfTasks - (i + 1), 1);
      queue.delete(getTask.getTaskId());
      counter = queue.getTaskCounter();
      assertCounter(counter, queueName, noOfTasks - (i + 1), 0);
    }
    assertTrue(queue.hasQualifiedTasks()); // no tasks, but qualifiers are still there.
    assertNull(queue.get(null, null));
  }

  /**
   * test rollback after exceeded time-to-live.
   * 
   * @throws Exception
   *           test fails
   */
  public void testGetTimedOutTasks() throws Exception {
    final String queueName = "testGetTimedOutTasks";
    final ZkTaskQueue queue = createQueue(queueName);
    TaskCounter counter = null;
    final int timeToLive = 1000;
    final int noOfTasks = 4;
    final Task[] putTasks = new Task[noOfTasks];
    for (int i = 0; i < noOfTasks; ++i) {
      putTasks[i] = createTask(i, queueName);
      queue.put(putTasks[i]);
      counter = queue.getTaskCounter();
      assertCounter(counter, queueName, i + 1, 0);
    }
    final Task[] getTasks = new Task[noOfTasks];
    for (int i = 0; i < noOfTasks; ++i) {
      getTasks[i] = queue.get(null, null);
      assertNotNull(getTasks[i]);
      counter = queue.getTaskCounter();
      assertCounter(counter, queueName, noOfTasks - (i + 1), i + 1);
    }
    Thread.sleep(2 * timeToLive);
    queue.keepAlive(getTasks[0].getTaskId());
    Collection<String> taskIds = queue.getTimedOutTasks(timeToLive);
    assertEquals(noOfTasks - 1, taskIds.size());
    assertFalse(taskIds.contains(getTasks[0].getTaskId()));
    for (final String taskId : taskIds) {
      queue.getInProgressTask(taskId);
      queue.delete(taskId);
    }
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 1);
    queue.getInProgressTask(getTasks[0].getTaskId());
    queue.delete(getTasks[0].getTaskId());
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 0);
    taskIds = queue.getTimedOutTasks(timeToLive);
    assertTrue(taskIds.isEmpty());
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 0);
  }

  /**
   * try to test parallel get requests.
   * 
   * @throws Exception
   *           test fails.
   */
  public void testParallelGet() throws Exception {
    final String queueName = "testParallelGet";
    final ZkTaskQueue queue = createQueue(queueName);
    TaskCounter counter = null;
    final int noOfTasks = 100;
    final int noOfThreads = 10;
    final Task[] putTasks = new Task[noOfTasks];
    for (int i = 0; i < noOfTasks; ++i) {
      putTasks[i] = createTask(i, queueName);
      queue.put(putTasks[i]);
    }
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, noOfTasks, 0);
    final Thread[] getThreads = new Thread[noOfThreads];
    final Exception[] getExceptions = new Exception[noOfThreads];
    for (int i = 0; i < noOfThreads; i++) {
      final int threadNo = i;
      getThreads[i] = new Thread(new Runnable() {
        @Override
        public void run() {
          try {
            while (true) {
              final Task getTask = queue.get(null, null);
              if (getTask == null) {
                return;
              }
              final int taskNo = Integer.parseInt(getTask.getTaskId());
              assertEqualTasks(putTasks[taskNo], getTask);
              queue.getInProgressTask(getTask.getTaskId());
              queue.delete(getTask.getTaskId());
              putTasks[taskNo] = null;
            }
          } catch (final Exception ex) {
            getExceptions[threadNo] = ex;
          }
        }
      }, "testParallelGet-Thread #" + threadNo);
    }
    for (int i = 0; i < noOfThreads; i++) {
      getThreads[i].start();
    }
    for (int i = 0; i < noOfThreads; i++) {
      getThreads[i].join();
      assertNull("Error in thread " + (i + 1) + ": " + getExceptions[i], getExceptions[i]);
    }
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 0);
  }

  /**
   * try to test parallel put requests.
   * 
   * @throws Exception
   *           test fails.
   */
  public void testParallelPut() throws Exception {
    final String queueName = "testParallelPut";
    final ZkTaskQueue queue = createQueue(queueName);
    TaskCounter counter = null;
    final int noOfTasksPerThread = 10;
    final int noOfThreads = 10;
    final int noOfTasks = noOfTasksPerThread * noOfThreads;
    final Task[] putTasks = new Task[noOfTasks];
    final Thread[] putThreads = new Thread[noOfThreads];
    final Exception[] putExceptions = new Exception[noOfThreads];
    for (int i = 0; i < noOfThreads; i++) {
      final int threadNo = i;
      putThreads[i] = new Thread(new Runnable() {
        @Override
        public void run() {
          try {
            for (int j = 0; j < noOfTasksPerThread; j++) {
              final int taskNo = (threadNo * noOfTasksPerThread) + j;
              putTasks[taskNo] = createTask(taskNo, queueName);
              queue.put(putTasks[taskNo]);
            }
          } catch (final Exception ex) {
            putExceptions[threadNo] = ex;
          }

        }
      }, "testParallelPut-Thread #" + threadNo);
    }
    for (int i = 0; i < noOfThreads; i++) {
      putThreads[i].start();
    }
    for (int i = 0; i < noOfThreads; i++) {
      putThreads[i].join();
      assertNull("Error in thread " + (i + 1) + ": " + putExceptions[i], putExceptions[i]);
    }
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, noOfTasks, 0);
    for (int i = 0; i < noOfTasks; i++) {
      final Task getTask = queue.get(null, null);
      assertNotNull(getTask);
      final int taskNo = Integer.parseInt(getTask.getTaskId());
      assertEqualTasks(putTasks[taskNo], getTask);
      queue.getInProgressTask(getTask.getTaskId());
      queue.delete(getTask.getTaskId());
    }
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 0);
  }

  /**
   * 
   * test purge.
   * 
   * @throws Exception
   *           test fails
   */
  public void testPurge() throws Exception {
    final String queueName = "testPurge";
    final ZkTaskQueue queue = createQueue(queueName);
    final Task task1 = createTask(1, queueName);
    queue.put(task1);
    final Task task2 = createTask(2, queueName);
    queue.put(task2);
    final Task task3 = createTask(3, queueName);
    task3.setQualifier("parts/partition");
    queue.put(task3);
    final Task task4 = createTask(4, queueName);
    task4.setQualifier("parts/partition");
    queue.put(task4);
    TaskCounter counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 4, 0);
    final Task getTask1 = queue.get(null, null);
    assertNotNull(getTask1);
    final Task getTask3 = queue.get("parts/partition", null);
    assertNotNull(getTask3);
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 2, 2);
    queue.purge();
    assertNull(queue.get(null, null));
    assertNull(queue.get("parts/partition", null));
    try {
      queue.delete(getTask1.getTaskId());
      fail("should not work");
    } catch (final BadParameterTaskmanagerException ex) {
      assertEquals(Cause.taskId, ex.getCauseCode());
    }
    try {
      queue.delete(getTask3.getTaskId());
      fail("should not work");
    } catch (final BadParameterTaskmanagerException ex) {
      assertEquals(Cause.taskId, ex.getCauseCode());
    }
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 0);
  }

  /**
   * check cleanup of unused partition queues.
   * 
   * @throws Exception
   *           test fails
   */
  public void testCleanQualifiedQueues() throws Exception {
    final String queueName = "testCleanQualifiedQueues";
    final ZkTaskQueue queue = createQueue(queueName);
    assertFalse(queue.hasQualifiedTasks());
    final ZooKeeperService zkService = getService(ZooKeeperService.class);
    assertNotNull(zkService);
    final ZkConnection zk = new ZkConnection(zkService);
    final Task task1 = createTask(1, queueName);
    task1.setQualifier("parts/partition1");
    final Task task2 = createTask(2, queueName);
    task2.setQualifier("parts/partition2");
    queue.put(task1);
    queue.put(task2);
    assertTrue(queue.hasQualifiedTasks());
    final String todoPartDir = ZkTaskQueue.TASKDIR_PREFIX + '/' + queueName + ZkTaskQueue.TODOQUALIFIEDDIR_SUFFIX;
    getChildrenChecked(zk, todoPartDir, 2);
    queue.cleanEmptyNodes(0); // should not cleanup anything, because there are still tasks.
    getChildrenChecked(zk, todoPartDir, 2);
    assertNotNull(queue.get("parts/partition1", null));
    assertNotNull(queue.get("parts/partition2", null));
    Thread.sleep(1000);
    final Task task3 = createTask(3, queueName);
    task3.setQualifier("parts/partition1");
    queue.put(task3);
    assertNotNull(queue.get("parts/partition1", null));
    final List<String> partitionDirs = getChildrenChecked(zk, todoPartDir, 2);
    queue.cleanEmptyNodes(500); // should remove partitions     
    assertEquals("parts_partition1", partitionDirs.get(0));
    assertTrue(queue.hasQualifiedTasks());
    Thread.sleep(50);
    queue.cleanEmptyNodes(1); // should remove partition1, too
    getChildrenChecked(zk, todoPartDir, 0);
    assertFalse(queue.hasQualifiedTasks());
  }

  /**
   * check removal of state tasks.
   * 
   * @throws Exception
   *           test fails.
   */
  public void testStaleTasks() throws Exception {
    final String queueName = "testStaleTasks";
    final ZkTaskQueue queue = createQueue(queueName);
    final ZooKeeperService zkService = getService(ZooKeeperService.class);
    assertNotNull(zkService);
    final ZkConnection zk = new ZkConnection(zkService);
    final Task putTask = createTask(1, queueName);
    queue.put(putTask);
    final String todoDir = ZkTaskQueue.TASKDIR_PREFIX + '/' + queueName + ZkTaskQueue.TODODIR_SUFFIX;
    final String staleTask = todoDir + "/staletask";
    zk.createNode(staleTask, ZkConnection.NO_DATA);
    TaskCounter counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 2, 0);
    queue.cleanEmptyNodes(1000);
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 2, 0);
    Thread.sleep(1000);
    queue.cleanEmptyNodes(1);
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 1, 0);
    final Task getTask = queue.get(null, null);
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 1);
    assertEqualTasks(putTask, getTask);
    queue.getInProgressTask(getTask.getTaskId());
    queue.delete(getTask.getTaskId());
    counter = queue.getTaskCounter();
    assertCounter(counter, queueName, 0, 0);
  }

  /** test task counters. */
  public void testTaskInfos() throws Exception {
    final String queueName = "testTaskInfos";
    final ZkTaskQueue queue = createQueue(queueName);
    final int noOfTasks = 4;
    final Task[] tasks = new Task[4];
    for (int i = 0; i < noOfTasks; i++) {
      tasks[i] = createTask(i, queueName);
      if (i >= noOfTasks / 2) {
        tasks[i].setQualifier("parts/partition");
      }
      queue.put(tasks[i]);
    }
    final TaskCounter todoCounter = queue.getTaskCounter();
    assertCounter(todoCounter, queueName, noOfTasks, 0);
    final TaskList startIpList = queue.getTaskList("inprogress", 1);
    assertEquals(0, startIpList.getSize());
    assertTrue(startIpList.getTaskNames().isEmpty());
    final TaskList todoTaskList = queue.getTaskList("todo", 10);
    assertEquals(noOfTasks, todoTaskList.getSize());
    final List<String> todoTaskNames = todoTaskList.getTaskNames();
    for (int i = 0; i < noOfTasks; i++) {
      final String taskPath = todoTaskNames.get(i);
      assertTrue(taskPath.startsWith("/smila/tasks/testTaskInfos/"));
      final String taskName = taskPath.substring("/smila/tasks/testTaskInfos/".length());
      if (i < noOfTasks / 2) {
        assertTrue(taskName.startsWith("todo/"));
      } else {
        assertTrue(taskName.startsWith("todo_qualified/parts_partition"));
      }
      final int index = taskName.indexOf('/');
      final AnyMap taskInfo = queue.getTaskInfo(taskName.substring(0, index), taskName.substring(index + 1));
      assertNotNull(taskInfo);
      assertTrue(taskInfo.containsKey("created"));
      assertTrue(taskInfo.containsKey("modified"));
      assertTrue(taskInfo.containsKey("task"));
      assertTrue(tasks[i].toAny().equals(taskInfo.get("task")));
      if (i < noOfTasks / 2) {
        queue.get(null, null);
      } else {
        queue.get("parts/partition", null);
      }
    }
    final TaskCounter ipCounter = queue.getTaskCounter();
    assertCounter(ipCounter, queueName, 0, noOfTasks);
    final TaskList endTodoList = queue.getTaskList("todo", 1);
    assertEquals(0, endTodoList.getSize());
    assertTrue(endTodoList.getTaskNames().isEmpty());
    final TaskList ipTaskList = queue.getTaskList("inprogress", 10);
    assertEquals(noOfTasks, ipTaskList.getSize());
    final List<String> ipTaskNames = ipTaskList.getTaskNames();
    for (int i = 0; i < noOfTasks; i++) {
      final String taskPath = ipTaskNames.get(i);
      assertTrue(taskPath.startsWith("/smila/tasks/testTaskInfos/"));
      final String taskName = taskPath.substring("/smila/tasks/testTaskInfos/".length());
      assertTrue(taskName.startsWith("inprogress/"));
      final int index = taskName.indexOf('/');
      final AnyMap taskInfo = queue.getTaskInfo(taskName.substring(0, index), taskName.substring(index + 1));
      assertNotNull(taskInfo);
      assertTrue(taskInfo.containsKey("created"));
      assertTrue(taskInfo.containsKey("modified"));
      assertTrue(taskInfo.containsKey("task"));
      final Task infoTask = Task.fromAny(taskInfo.get("task"));
      infoTask.getProperties().remove(ZkTaskQueue.OWNER_PROP);
      //remove start and end time
      infoTask.getProperties().remove(Task.PROPERTY_END_TIME);
      infoTask.getProperties().remove(Task.PROPERTY_START_TIME);
      infoTask.getProperties().remove(Task.PROPERY_TASK_AGE);
      assertTrue(tasks[i].toAny().equals(infoTask.toAny()));
      queue.getInProgressTask(infoTask.getTaskId());
      queue.delete(infoTask.getTaskId());
    }
    final TaskCounter endCounter = queue.getTaskCounter();
    assertCounter(endCounter, queueName, 0, 0);
    final TaskList endIpList = queue.getTaskList("inprogress", 1);
    assertEquals(0, endIpList.getSize());
    assertTrue(endIpList.getTaskNames().isEmpty());
  }

  /**
   * test that 'host' property is set.
   */
  public void testHostProperty() throws Exception {
    final String queueName = "testHost";
    final ZkTaskQueue queue = createQueue(queueName);
    // without host
    Task putTask = createTask(1, queueName);
    queue.put(putTask);
    Task getTask = queue.get(null, null);
    assertNotNull(getTask);
    assertNull(getTask.getProperties().get(Task.PROPERTY_WORKER_HOST));
    Task ipTask = queue.getInProgressTask(getTask.getTaskId());
    assertNotNull(ipTask);
    assertNull(ipTask.getProperties().get(Task.PROPERTY_WORKER_HOST));
    assertEqualTasks(getTask, ipTask);

    // with host
    putTask = createTask(2, queueName);
    queue.put(putTask);
    getTask = queue.get(null, "host");
    assertNotNull(getTask);
    assertEquals("host", getTask.getProperties().get(Task.PROPERTY_WORKER_HOST));
    ipTask = queue.getInProgressTask(getTask.getTaskId());
    assertNotNull(ipTask);
    assertNull(ipTask.getProperties().get(Task.PROPERTY_WORKER_HOST));
    getTask.getProperties().remove(Task.PROPERTY_WORKER_HOST);
    assertEqualTasks(getTask, ipTask);
  }

  /** test if task delivery is limited. */
  public void testScaleUpControl() throws Exception {
    final String queueName = "testScaleUpControl";
    final String hostName = queueName;
    final ZkTaskQueue queue = createQueue(queueName, 2);
    queue.put(createTask(1, queueName));
    queue.put(createTask(2, queueName));
    queue.put(createTask(3, queueName));
    final Task task1 = queue.get(null, hostName);
    assertNotNull(task1);
    final Task task2 = queue.get(null, hostName);
    assertNotNull(task2);
    Task task3 = queue.get(null, hostName);
    assertNull(task3);
    queue.getInProgressTask(task1.getTaskId());
    queue.delete(task1.getTaskId());
    task3 = queue.get(null, hostName);
    assertNotNull(task3);
    queue.getInProgressTask(task2.getTaskId());
    queue.delete(task2.getTaskId());
    queue.getInProgressTask(task3.getTaskId());
    queue.delete(task3.getTaskId());
  }

  /** test if scaleup control does not affect runAlways workers. */
  public void testNoLimitForRunAlways() throws Exception {
    final String queueName = "testLimitForRunAlways";
    final String hostName = null;
    final ZkTaskQueue queue = createQueue(queueName, 2);
    queue.put(createTask(1, queueName));
    queue.put(createTask(2, queueName));
    queue.put(createTask(3, queueName));
    final Task task1 = queue.get(null, hostName);
    assertNotNull(task1);
    final Task task2 = queue.get(null, hostName);
    assertNotNull(task2);
    final Task task3 = queue.get(null, hostName);
    assertNotNull(task3);
    queue.getInProgressTask(task1.getTaskId());
    queue.delete(task1.getTaskId());
    queue.getInProgressTask(task2.getTaskId());
    queue.delete(task2.getTaskId());
    queue.getInProgressTask(task3.getTaskId());
    queue.delete(task3.getTaskId());
  }

  /**
   * get children of path and check for count.
   * 
   * @param zk
   *          ZooKeeper connection
   * @param path
   *          path
   * @param expectedCount
   *          expected number of children
   * @return children
   * @throws Exception
   *           fail
   */
  private List<String> getChildrenChecked(final ZkConnection zk, final String path, final int expectedCount)
    throws Exception {
    final List<String> partitionDirs = zk.getChildrenSorted(path);
    assertEquals(expectedCount, partitionDirs.size());
    return partitionDirs;
  }

  /**
   * create queue.
   * 
   * @param queueName
   *          name
   * @return queue
   */
  private ZkTaskQueue createQueue(final String queueName) {
    return createQueue(queueName, TaskStorageZk.TASKSPERHOST_UNLIMITED);
  }

  /**
   * create queue with scale-up limit.
   */
  private ZkTaskQueue createQueue(final String queueName, final long maxScaleUp) {
    return new ZkTaskQueue(_zk, queueName, "localhost", maxScaleUp);
  }

  /**
   * @return created task
   */
  protected static Task createTask(final int taskNo, final String workerName) {
    final String taskId = Integer.toString(taskNo);
    final Task task = new Task(taskId, workerName);
    task.getProperties().put("worker", workerName);
    task.getProperties().put("taskId", taskId);
    task.getParameters().put("worker", workerName);
    task.getParameters().put("taskId", taskId);
    final List<BulkInfo> input = new ArrayList<BulkInfo>();
    input.add(new BulkInfo("inputbucket", "inputstore", "input" + taskId));
    task.getInputBulks().put("input", input);
    final List<BulkInfo> output = new ArrayList<BulkInfo>();
    output.add(new BulkInfo("outputbucket", "outputstore", "output" + taskId));
    task.getInputBulks().put("output", output);
    return task;
  }

  /**
   * assert that all task members and properties are equal.
   * 
   * @param putTask
   *          expected task
   * @param getTask
   *          actual task
   */
  protected static void assertEqualTasks(final Task putTask, final Task getTask) {
    assertEquals(putTask.getTaskId(), getTask.getTaskId());
    assertEquals(putTask.getWorkerName(), getTask.getWorkerName());
    assertEquals(putTask.getQualifier(), getTask.getQualifier());
    //remove start and end time
    getTask.getProperties().remove(Task.PROPERTY_START_TIME);
    getTask.getProperties().remove(Task.PROPERTY_END_TIME);
    getTask.getProperties().remove(Task.PROPERY_TASK_AGE);
    putTask.getProperties().remove(Task.PROPERTY_START_TIME);
    putTask.getProperties().remove(Task.PROPERTY_END_TIME);
    putTask.getProperties().remove(Task.PROPERY_TASK_AGE);
    assertEquals(putTask.getProperties(), getTask.getProperties());
    assertEquals(putTask.getParameters(), getTask.getParameters());
    assertEquals(putTask.getInputBulks(), getTask.getInputBulks());
    assertEquals(putTask.getOutputBulks(), getTask.getOutputBulks());
  }

  /**
   * assert task counter values.
   * 
   * @param counter
   *          actual counter
   * @param name
   *          expected name
   * @param todo
   *          expected todo count
   * @param inProgress
   *          expected inprogress count.
   */
  protected static void assertCounter(final TaskCounter counter, final String name, final int todo,
    final int inProgress) {
    assertNotNull(counter);
    assertEquals("wrong name", name, counter.getWorkerName());
    assertEquals("wrong todo counter", todo, counter.getTasksTodo());
    assertEquals("wrong inprogress counter", inProgress, counter.getTasksInProgress());
  }

}
