/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.httphandler;

import java.net.MalformedURLException;
import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;

import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskManager;

/**
 * Implements the handling of HTTP requests for the task state API.
 * 
 * URL pattern: smila/tasks/
 */
public class TaskStateHandler extends JsonRequestHandler {

  /** The reference to the TaskManager service. */
  private TaskManager _taskManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("GET".equals(method)) {
      final AnyMap result = FACTORY.createAnyMap();
      result.put("failsafety", _taskManager.getFailSafetyLevel());
      final Map<String, TaskCounter> taskCounters = _taskManager.getTaskCounters();
      final AnySeq workers = FACTORY.createAnySeq();
      for (final TaskCounter counter : taskCounters.values()) {
        final AnyMap workerSection = FACTORY.createAnyMap();
        workerSection.put("name", counter.getWorkerName());
        createCounterSection(workerSection, requestUri, "todo", counter.getTasksTodo());
        createCounterSection(workerSection, requestUri, "inprogress", counter.getTasksInProgress());
        workers.add(workerSection);
      }
      result.put("workers", workers);
      final AnySeq scaleUp = FACTORY.createAnySeq();
      final Map<String, Integer> scaleUpCounters = _taskManager.getScaleUpCounters();
      for (final Map.Entry<String, Integer> counter : scaleUpCounters.entrySet()) {
        final AnyMap hostSection = FACTORY.createAnyMap();
        hostSection.put("host", counter.getKey());
        hostSection.put("count", counter.getValue());
        scaleUp.add(hostSection);
      }
      result.put("maxScaleUp", _taskManager.getMaxScaleUp());
      result.put("scaleUp", scaleUp);
      return result;
    }
    return null;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalArgumentException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex.getCause() != null && ex.getCause() instanceof BadParameterTaskmanagerException) {
      return HttpStatus.NOT_FOUND;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * add section for a single worker task type (e.g. "todo", "inprogress") to the task pipe section. Expects that
   * taskPipeSection contains the pipe name already.
   * 
   * @param workerTaskSection
   *          worker task section
   * @param requestUri
   *          base request URI
   * @param type
   *          type of queue part
   * @param count
   *          number of tasks in queue part.
   * @throws MalformedURLException
   *           cannot create URL - the problem is probably the task pipe or type name.
   */
  private void createCounterSection(final AnyMap workerTaskSection, final String requestUri, final String type,
    final long count) throws MalformedURLException {
    final AnyMap counterSection = FACTORY.createAnyMap();
    counterSection.put("count", count);
    counterSection.put("url",
      URLCreator.create(getRequestHost(), requestUri, workerTaskSection.get("name") + "/" + type).toString());
    workerTaskSection.put(type, counterSection);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and DELETE are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return ("GET".equals(method));
  }

  /**
   * @param taskManager
   *          the reference to the JobManager service.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * @param taskManager
   *          the reference to the current JobManager service.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }
}
