/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.taskworker.test;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.objectstore.NoSuchObjectException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreOutputStream;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.input.Input;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.input.StreamInput;
import org.eclipse.smila.utils.collections.MultiValueMap;

/** unit tests for the input wrappers. */
public class TestInputs extends InputOutputTestbase {

  protected static final String STORE_NAME = "inputs";

  protected Inputs _inputs;

  /** does nothing, just to test instantiation. */
  public static class CustomInput extends Input {

    /** input objects must provide this constructor. */
    public CustomInput(final BulkInfo dataObject, final ObjectStoreService objectStore) {
      super(dataObject, objectStore);
    }

    @Override
    public void close() {
      // nothing to do
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();    
    _service.removeStore("inputs");
    _service.createStore("inputs", null);
    _inputs = new Inputs(prepareDataObjects(), _service);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void tearDown() throws Exception {
    _inputs.closeAll();
    super.tearDown();
  }

  /**
   * Tests if added data is existing.
   * 
   * @throws Exception
   */
  public void testGetDataObject() throws Exception {
    assertEquals(1, _inputs.getDataObjectCount("streamSingle"));
    final BulkInfo streamSingle = _inputs.getDataObject("streamSingle");
    assertNotNull(streamSingle);
    assertEquals("streamSingle", streamSingle.getObjectName());
    assertSame(streamSingle, _inputs.getDataObject("streamSingle", 0));
    assertEquals(3, _inputs.getDataObjectCount("streamMulti"));
    final BulkInfo streamMulti0 = _inputs.getDataObject("streamMulti");
    assertNotNull(streamMulti0);
    assertEquals("streamMulti0", streamMulti0.getObjectName());
    assertSame(streamMulti0, _inputs.getDataObject("streamMulti", 0));
    assertEquals("streamMulti1", _inputs.getDataObject("streamMulti", 1).getObjectName());
    assertEquals("streamMulti2", _inputs.getDataObject("streamMulti", 2).getObjectName());
  }

  /**
   * Tests if the added strings are are accessible with the {@link StreamInput} helper.
   * 
   * @throws Exception
   */
  public void testStreamInput() throws Exception {
    _service.putObject(STORE_NAME, "streamSingle", "abc".getBytes());
    final StreamInput streamSingle = _inputs.getAsStreamInput("streamSingle");
    assertEquals("abc", IOUtils.toString(streamSingle.getStream()));
    streamSingle.close();
    assertEquals(3, streamSingle.getBytesRead());
    _service.putObject(STORE_NAME, "streamMulti1", "123".getBytes());
    final StreamInput streamMulti0 = _inputs.getAsStreamInput("streamMulti", 0);
    try {
      streamMulti0.getStream();
      fail("Should not happen, object streamMulti does not exist.");
    } catch (final ObjectStoreException ex) {
      assertTrue(ex instanceof NoSuchObjectException);
    }
    final StreamInput streamMulti1 = _inputs.getAsStreamInput("streamMulti", 1);
    assertEquals("123", IOUtils.toString(streamMulti1.getStream()));
    streamMulti1.close();
    assertEquals(3, streamMulti1.getBytesRead());
  }

  /**
   * Tests if the added records are are accessible as records with the {@link RecordInput} helper.
   * 
   * @throws Exception
   */
  public void testRecordInput() throws Exception {
    final StoreOutputStream recordStream = _service.writeObject(STORE_NAME, "recordSingle");
    recordStream.write(createRecordBon("record0"));
    recordStream.write(createRecordBon("record1"));
    recordStream.write(createRecordBon("record2"));
    recordStream.close();
    final RecordInput recordSingle = _inputs.getAsRecordInput("recordSingle");
    try {
      final Record r0 = recordSingle.getRecord();
      assertEquals("record0", r0.getId());
      assertEquals("record0", r0.getMetadata().getStringValue("title"));
      final Record r1 = recordSingle.getRecord();
      assertEquals("record1", r1.getId());
      assertEquals("record1", r1.getMetadata().getStringValue("title"));
      final Record r2 = recordSingle.getRecord();
      assertEquals("record2", r2.getId());
      assertEquals("record2", r2.getMetadata().getStringValue("title"));
      assertNull(recordSingle.getRecord());
      assertEquals(3, recordSingle.getRecordCount());
      assertTrue(recordSingle.getBytesRead() > 0);
      try {
        _inputs.getAsRecordInput("recordMulti", 2).getRecord();
        fail("Should not happen, object recordMulti does not exist.");
      } catch (final ObjectStoreException ex) {
        assertTrue(ex instanceof NoSuchObjectException);
      }
    } finally {
      recordSingle.close();
    }
  }

  /** test instantiation of a custom input class. */
  public void testCustomInput() throws Exception {
    final CustomInput custom = _inputs.getAsInput("streamSingle", CustomInput.class);
    assertNotNull(custom);
    assertEquals(STORE_NAME, custom.getStoreName());
    assertEquals("streamSingle", custom.getObjectName());
  }

  /** setup input data object map. */
  private Map<String, List<BulkInfo>> prepareDataObjects() {
    final MultiValueMap<String, BulkInfo> dataObjects = new MultiValueMap<String, BulkInfo>();
    dataObjects.add("streamSingle", new BulkInfo("x", "inputs", "streamSingle"));
    dataObjects.add("streamMulti", new BulkInfo("x", "inputs", "streamMulti0"));
    dataObjects.add("streamMulti", new BulkInfo("x", "inputs", "streamMulti1"));
    dataObjects.add("streamMulti", new BulkInfo("x", "inputs", "streamMulti2"));
    dataObjects.add("recordSingle", new BulkInfo("x", "inputs", "recordSingle"));
    dataObjects.add("recordMulti", new BulkInfo("x", "inputs", "recordMulti0"));
    dataObjects.add("recordMulti", new BulkInfo("x", "inputs", "recordMulti1"));
    dataObjects.add("recordMulti", new BulkInfo("x", "inputs", "recordMulti2"));
    dataObjects.add("kvoSingle", new BulkInfo("x", "inputs", "kvoSingle.kvo"));
    dataObjects.add("kvoMulti", new BulkInfo("x", "inputs", "kvoSingle0.kvo"));
    dataObjects.add("kvoMulti", new BulkInfo("x", "inputs", "kvoSingle1.kvo"));
    dataObjects.add("kvoMulti", new BulkInfo("x", "inputs", "kvoSingle2.kvo"));
    return dataObjects;
  }

  /**
   * @param title
   *          The title
   * @return a record with the given title as byte array
   * @throws IOException
   */
  protected byte[] createRecordBon(final String title) throws IOException {
    final Record r = DataFactory.DEFAULT.createRecord(title);
    r.getMetadata().put("title", title);
    return IPC_RECORD_WRITER.writeBinaryObject(r);
  }
}
