/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.taskworker.test.worker;

import java.lang.reflect.Constructor;

import org.eclipse.smila.datamodel.Record;

import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.Outputs;
import org.eclipse.smila.taskworker.output.RecordOutput;
import org.eclipse.smila.taskworker.test.TestInputs.CustomInput;
import org.eclipse.smila.taskworker.test.TestOutputs.CustomOutput;

/**
 * Test worker that just throws an exceptions once (or forever, depending on the configuration) and then starts copying
 * records to output.
 * 
 * The exception to throw can be either set by task parameter or directly via (static) setter method.
 */
public class ExceptionalWorker implements Worker {

  public static final String NAME = "exceptionalWorker";

  public static final String PARAM_EXCEPTION_TO_THROW = "exceptionToThrow";

  public static final String PARAM_EXCEPTION_THROWING_FOREVER = "exceptionThrowingForever";

  public static final String PARAM_EXCEPTION_MESSAGE = "exceptionMessage";

  private static final String INPUT_SLOT_NAME = "exceptionalInput";

  private static final String OUTPUT_SLOT_NAME = "exceptionalOutput";

  private static Class<? extends Exception> s_classOfThrowableToThrow;

  private static boolean s_exceptionThrowingForever;

  private static String s_exceptionMessage;

  /**
   * default constructor used when getting OSGI service instance.
   */
  public ExceptionalWorker() {
  }

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    final Task task = taskContext.getTask();
    if (task != null && task.getParameters() != null) {
      if (task.getParameters().containsKey(PARAM_EXCEPTION_TO_THROW)) {
        final String ett = task.getParameters().getStringValue(PARAM_EXCEPTION_TO_THROW);
        setExceptionToThrow(ett);
      }
      if (task.getParameters().containsKey(PARAM_EXCEPTION_MESSAGE)) {
        final String em = task.getParameters().getStringValue(PARAM_EXCEPTION_MESSAGE);
        setExceptionMessage(em);
      }
      if (task.getParameters().containsKey(PARAM_EXCEPTION_THROWING_FOREVER)) {
        final Boolean etf = Boolean.valueOf(task.getParameters().getStringValue(PARAM_EXCEPTION_THROWING_FOREVER));
        setExceptionThrowingForever(etf.booleanValue());
      }
    }
    final Inputs inputs = taskContext.getInputs();
    final Outputs outputs = taskContext.getOutputs();
    if (s_classOfThrowableToThrow != null) {
      // just a test if instantiation works for classes defined in user bundles.
      final CustomInput input = inputs.getAsInput(INPUT_SLOT_NAME, CustomInput.class);
      final CustomOutput output = outputs.getAsOutput(OUTPUT_SLOT_NAME, CustomOutput.class);
      if (input != null && output != null) {
        try {
          final Constructor<? extends Exception> ctor = s_classOfThrowableToThrow.getConstructor(String.class);
          ctor.setAccessible(true);
          throw ctor.newInstance(s_exceptionMessage);
        } finally {
          if (!s_exceptionThrowingForever) {
            s_classOfThrowableToThrow = null;
          }
        }
      }
    } else {
      final RecordInput recordInput = inputs.getAsRecordInput(INPUT_SLOT_NAME);
      final RecordOutput recordOutput = outputs.getAsRecordOutput(OUTPUT_SLOT_NAME);
      Record record = recordInput.getRecord();
      while (record != null) {
        recordOutput.writeRecord(record);
        record = recordInput.getRecord();
      }
    }
  }

  @Override
  public String getName() {
    return NAME;
  }

  /**
   * @param ett
   *          the exception to throw in perform(), null or empty if no exception shall be thrown.
   * @throws ClassNotFoundException
   */
  @SuppressWarnings("unchecked")
  public static void setExceptionToThrow(final String ett) throws ClassNotFoundException {
    if (ett != null && !"".equals(ett)) {
      s_classOfThrowableToThrow = (Class<? extends Exception>) Class.forName(ett);
    } else {
      s_classOfThrowableToThrow = null;
    }
  }

  /**
   * En- or disable repeatedly throwing excetions. If set to false, an exception is thrown only once, if set to true,
   * exceptions are thrown at each invocation of perform(...).
   * 
   * @param booleanValue
   *          If false, an exception is thrown only once, if true, exceptions are thrown at each invocation of
   *          perform(...).
   */
  public static void setExceptionThrowingForever(final boolean booleanValue) {
    s_exceptionThrowingForever = booleanValue;
  }

  /**
   * Set the message of the exception to throw.
   * 
   * @param message
   *          The exception's message.
   */
  public static void setExceptionMessage(final String message) {
    s_exceptionMessage = message;
  }

}
