/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.utils;

/**
 * base class for exception that have an isRecoverable flag. If the flag is true, it means that the exception condition
 * is probably not permanent and the caller is encouraged to retry the request, maybe after a short delay time. Examples
 * for such reasons could be external resources that are currently not reachable, network problems and similar.
 * Non-recoverable are invalid arguments, parse errors, external resources that really do not exists, etc.
 * 
 */
public class MaybeRecoverableException extends Exception {

  /** is serializable. */
  private static final long serialVersionUID = 1L;

  /** recoverable exception? */
  private final boolean _recoverable;

  /**
   * create exception. recoverable flag is inherited from cause, if cause is a {@link MaybeRecoverableException}, too.
   * Else the flag is set to false.
   */
  public MaybeRecoverableException(final String message, final Throwable cause) {
    this(message, cause, isRecoverableCause(cause));
  }

  /**
   * create exception. recoverable flag is set to false.
   */
  public MaybeRecoverableException(final String message) {
    this(message, false);
  }

  /**
   * create exception. recoverable flag is inherited from cause, if cause is a {@link MaybeRecoverableException}, too.
   * Else the flag is set to false.
   */
  public MaybeRecoverableException(final Throwable cause) {
    this(cause, isRecoverableCause(cause));
  }

  /**
   * create exception, recoverable flag explicitly defined.
   */
  public MaybeRecoverableException(final String message, final Throwable cause, final boolean recoverable) {
    super(message, cause);
    _recoverable = recoverable;
  }

  /**
   * create exception, recoverable flag explicitly defined.
   */
  public MaybeRecoverableException(final String message, final boolean recoverable) {
    super(message);
    _recoverable = recoverable;
  }

  /**
   * create exception, recoverable flag explicitly defined.
   */
  public MaybeRecoverableException(final Throwable cause, final boolean recoverable) {
    super(cause);
    _recoverable = recoverable;
  }

  /** @return if cause is a {@link MaybeRecoverableException}, too, return it's recoverable flag. else false. */
  public static boolean isRecoverableCause(final Throwable cause) {
    if (cause instanceof MaybeRecoverableException) {
      return ((MaybeRecoverableException) cause).isRecoverable();
    }
    return false;
  }

  /** @return true if is useful for the client to retry the operation. */
  public boolean isRecoverable() {
    return _recoverable;
  }
}
