/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.utils.conversion;

import java.util.HashMap;
import java.util.Map;

/**
 * parses strings describing data sizes to a long giving the number of bytes specified.
 */
public final class DataSizeParser {

  /** factors to calculate bytes from suffix. */
  private static final Map<Character, Long> SUFFIX_FACTOR = new HashMap<Character, Long>();

  /** supported suffixes. */
  private static final char[] SUFFIX = { 'b', 'k', 'm', 'g', 't', 'p' };

  /** bit shift to calculate suffix factors. */
  private static final long FACTOR = 10;

  static {
    long factor = 1;
    for (final Character suffix : SUFFIX) {
      SUFFIX_FACTOR.put(suffix, factor);
      factor = factor << FACTOR;
    }
  }

  /**
   * prevent instantiation.
   * 
   * @throws InstantiationException
   *           class cannot be instantiated
   */
  private DataSizeParser() throws InstantiationException {
    throw new InstantiationException("utility class only.");
  }

  /**
   * parse data size and return number of bytes described by the string. The string contains either a simple long value
   * giving the number of bytes immediately, or a double value with a single character suffix specifying the magnitude.
   * Supported suffixes and their meaning are:
   * <ul>
   * <li>b - bytes
   * <li>k - kibibyte (2^10 byte)
   * <li>m - mebibyte (2^20 byte)
   * <li>g - gibibyte (2^30 byte)
   * <li>t - tebibyte (2^40 byte)
   * <li>p - pebibyte (2^50 byte)
   * </ul>
   * 
   * E.g., "200m" would be 200 mebibyte, "1.5g" would be 1.5 gibibyte. Suffixes can be in upper case, too. If the input
   * string is empty, the defaultSize value is returned. If the string contains an unsupported suffix, an
   * {@link IllegalArgumentException} is thrown. If the number part cannot be parsed, a {@link NumberFormatException} is
   * thrown.
   * 
   * @param dataSize
   *          string in described format
   * @param defaultSize
   *          value to return, if dataSize is null or an empty string.
   * @return number of bytes described by the string.
   */
  public static long parse(final String dataSize, final long defaultSize) {
    if (dataSize == null) {
      return defaultSize;
    }
    final int length = dataSize.length();
    if (length == 0) {
      return defaultSize;
    }
    final char lastChar = dataSize.charAt(length - 1);
    if (lastChar >= '0' && lastChar <= '9') {
      return Long.parseLong(dataSize);
    } else {
      final Character suffix = Character.toLowerCase(lastChar);
      if (!SUFFIX_FACTOR.containsKey(suffix)) {
        throw new IllegalArgumentException("Unknown magnitude character '" + lastChar + "'");
      }
      final String numberPart = dataSize.substring(0, length - 1);
      try {
        final long number = Long.parseLong(numberPart);
        return number * SUFFIX_FACTOR.get(suffix);
      } catch (final NumberFormatException ex) {
        final double number = Double.parseDouble(numberPart);
        return (long) (number * SUFFIX_FACTOR.get(suffix));
      }
    }
  }

  /**
   * calls {@link #parse(String, long)} with defaultSize 0.
   * 
   * @param dataSize
   *          string in described format
   * @return number of bytes described by the string.
   */
  public static long parse(final String dataSize) {
    return parse(dataSize, 0);
  }
}
