/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager.test;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;

import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskList;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.workermanager.TaskKeepAliveListener;
import org.eclipse.smila.workermanager.keepalive.DelayedTask;
import org.eclipse.smila.workermanager.keepalive.TaskKeepAlive;

/**
 * Test class for TaskKeepAlive.
 */
public class TestTaskKeepAlive extends TestCase {
  /** the only allowed workerName. */
  private static final String WORKER_NAME = "workerName";

  /** executor service for TaskKeepAlive. */
  private ScheduledExecutorService _scheduledExecutor;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _scheduledExecutor = Executors.newSingleThreadScheduledExecutor();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void tearDown() throws Exception {
    _scheduledExecutor.shutdown();
  }

  /**
   * Dummy task manager service for testing.
   * 
   */
  private static class DummyTaskManager implements TaskManager, TaskKeepAliveListener {

    /** a list with tasks that have been kept alive. */
    private final List<String> _keptAliveTasks = new ArrayList<String>();

    /** a list of currently active tasks. */
    private final List<String> _taskList = new ArrayList<String>();

    /** allowed worker names. */
    private final List<String> _workerList = Collections.singletonList(WORKER_NAME);

    /** removed tasks. */
    private final Collection<Task> _removedTasks = new ArrayList<Task>();

    @Override
    public void addInProgressTask(final Task task) throws TaskmanagerException {
    }

    @Override
    public Task getTask(final String workerName, final String host) throws TaskmanagerException {
      return null;
    }

    @Override
    public Task getTask(final String workerName, final String host, final Collection<String> qualifiers)
      throws TaskmanagerException {
      return null;
    }

    /** {@inheritDoc} */
    @Override
    public void finishTask(final String workerName, final String taskId, final ResultDescription resultDescription)
      throws TaskmanagerException {
      _taskList.remove(taskId);
    }

    /** {@inheritDoc} */
    @Override
    public synchronized void keepAlive(final String workerName, final String taskId) throws TaskmanagerException {
      _keptAliveTasks.add(taskId);
      //System.out.println("kept alive: " + taskId);
      if (!_taskList.contains(taskId)) {
        throw new BadParameterTaskmanagerException("no such task", BadParameterTaskmanagerException.Cause.taskId);
      }
      if (!_workerList.contains(workerName)) {
        throw new BadParameterTaskmanagerException("no such worker",
          BadParameterTaskmanagerException.Cause.workerName);
      }
    }

    /** {@inheritDoc} */
    @Override
    public void finishTasks(final String workerName, final Collection<String> qualifiers,
      final ResultDescription resultDescription) throws TaskmanagerException {
    }

    /** {@inheritDoc} */
    @Override
    public Map<String, TaskCounter> getTaskCounters() throws TaskmanagerException {
      return null;
    }

    /** {@inheritDoc} */
    @Override
    public TaskList getTaskList(final String workerName, final String section, final int maxCount)
      throws TaskmanagerException {
      return null;
    }

    /** {@inheritDoc} */
    @Override
    public Any getTaskInfo(final String workerName, final String section, final String taskName)
      throws TaskmanagerException {
      return null;
    }

    /** {@inheritDoc} */
    @Override
    public long getFailSafetyLevel() {
      return 0;
    }

    @Override
    public long getMaxScaleUp() {
      return -1;
    }

    @Override
    public Map<String, Integer> getScaleUpCounters() throws TaskmanagerException {
      return new HashMap<String, Integer>();
    }

    /** {@inheritDoc} */
    @Override
    public void addTasks(final Collection<Task> taskList) throws TaskmanagerException {
    }

    /**
     * @return kept alive tasks
     */
    public List<String> getKeptAliveTasks() {
      return _keptAliveTasks;
    }

    /**
     * add a task.
     * 
     * @param addTask
     *          the task to add
     */
    @Override
    public synchronized void addTask(final Task addTask) {
      _taskList.add(addTask.getTaskId());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removedTask(final Task task) {
      _removedTasks.add(task);
    }

    /**
     * @return the _removedTasks
     */
    public Collection<Task> getRemovedTasks() {
      return _removedTasks;
    }

    /** {@inheritDoc} */
    @Override
    public void removeTasks(final AnyMap filterMap) throws TaskmanagerException {
    }

    @Override
    public void addTaskQueue(final String workerName) throws TaskmanagerException {
    }
  }

  /**
   * tests that tasks will be kept alive regularly and cease to be kept alive when removed.
   * 
   * @throws InterruptedException
   *           thread was being interrupted.
   * @throws TaskmanagerException
   *           any unexpected task manager exception.
   */
  public void testKeepTasksAlive() throws InterruptedException, TaskmanagerException {

    final int waitTime = 8000;

    // -- init
    final DummyTaskManager myManager = new DummyTaskManager();
    // 5 seconds timeout, 1 second between checks should be enough for testing
    final TaskKeepAlive taskKeepAlive = new TaskKeepAlive(myManager, 5);
    taskKeepAlive.addKeepAliveFailureListener(myManager);
    _scheduledExecutor.scheduleAtFixedRate(taskKeepAlive, 2, 2, TimeUnit.SECONDS);

    // -- add some tasks
    final int noOfTasks = 3;
    final Task task1 = new Task("1", WORKER_NAME);
    myManager.addTask(task1);
    taskKeepAlive.addTask(task1);
    final Task task2 = new Task("2", WORKER_NAME);
    myManager.addTask(task2);
    taskKeepAlive.addTask(task2);
    final Task task3 = new Task("3", WORKER_NAME);
    myManager.addTask(task3);
    taskKeepAlive.addTask(task3);

    // -- wait  
    Thread.sleep(waitTime);
    assertEquals(noOfTasks, myManager.getKeptAliveTasks().size());
    assertTrue(myManager.getKeptAliveTasks().contains(task1.getTaskId()));
    assertTrue(myManager.getKeptAliveTasks().contains(task2.getTaskId()));
    assertTrue(myManager.getKeptAliveTasks().contains(task3.getTaskId()));
    // clear list
    myManager.getKeptAliveTasks().clear();

    // -- remove first task
    taskKeepAlive.removeTask(task1);
    myManager.finishTask(task1.getWorkerName(), task1.getTaskId(), null);

    // -- add another
    final Task task4 = new Task("4", WORKER_NAME);
    myManager.addTask(task4);
    taskKeepAlive.addTask(task4);

    // -- wait
    Thread.sleep(waitTime);
    assertEquals("tasks: " + myManager.getKeptAliveTasks(), noOfTasks, myManager.getKeptAliveTasks().size());
    assertFalse(myManager.getKeptAliveTasks().contains(task1.getTaskId()));
    assertTrue(myManager.getKeptAliveTasks().contains(task2.getTaskId()));
    assertTrue(myManager.getKeptAliveTasks().contains(task3.getTaskId()));
    assertTrue(myManager.getKeptAliveTasks().contains(task4.getTaskId()));
    myManager.getKeptAliveTasks().clear();

    // -- remove all
    taskKeepAlive.removeTask(task2);
    taskKeepAlive.removeTask(task3);
    taskKeepAlive.removeTask(task4);
    myManager.finishTask(task1.getWorkerName(), task2.getTaskId(), null);
    myManager.finishTask(task1.getWorkerName(), task3.getTaskId(), null);
    myManager.finishTask(task1.getWorkerName(), task4.getTaskId(), null);

    // -- wait
    Thread.sleep(waitTime);
    assertEquals(0, myManager.getKeptAliveTasks().size());
    taskKeepAlive.removeKeepAliveFailureListener(myManager);
    assertTrue(myManager.getRemovedTasks().isEmpty());
  }

  /**
   * tests if startup and shutdown work correctly.
   * 
   * @throws InterruptedException
   *           test failed
   */
  public void testTaskKeepAliveStartupAndShutdown() throws InterruptedException {
    // -- init
    final DummyTaskManager myManager = new DummyTaskManager();
    // 2 seconds timeout, 1 second between checks should be enough for testing
    TaskKeepAlive taskKeepAlive = new TaskKeepAlive(myManager, 2);
    _scheduledExecutor.scheduleAtFixedRate(taskKeepAlive, 1, 1, TimeUnit.SECONDS);

    // -- add some tasks
    final Task task1 = new Task("1", WORKER_NAME);
    myManager.addTask(task1);
    taskKeepAlive.addTask(task1);
    final Task task2 = new Task("2", WORKER_NAME);
    myManager.addTask(task2);
    taskKeepAlive.addTask(task2);
    final Task task3 = new Task("3", WORKER_NAME);
    myManager.addTask(task3);
    taskKeepAlive.addTask(task3);

    // -- shutdown
    _scheduledExecutor.shutdown();

    // -- wait
    final int waitTime = 4000;
    Thread.sleep(waitTime);

    assertEquals(0, myManager.getKeptAliveTasks().size());

    // --startup again
    _scheduledExecutor = Executors.newSingleThreadScheduledExecutor();
    // we simulate shutdown and startup of BulkBuilderService, which also creates a new TaskKeepAlive...
    taskKeepAlive = new TaskKeepAlive(myManager, 2);
    _scheduledExecutor.scheduleAtFixedRate(taskKeepAlive, 1, 1, TimeUnit.SECONDS);

    // -- wait
    Thread.sleep(waitTime);
    // nevertheless the old added task will not have been kept alive
    assertEquals(0, myManager.getKeptAliveTasks().size());

    // -- add a new task
    final Task task4 = new Task("4", WORKER_NAME);
    myManager.addTask(task4);
    taskKeepAlive.addTask(task4);

    // -- wait again
    Thread.sleep(waitTime);
    // nevertheless the old added task will not have been kept alive
    assertTrue(myManager.getKeptAliveTasks().size() > 0);
    assertFalse(myManager.getKeptAliveTasks().contains(task1.getTaskId()));
    assertFalse(myManager.getKeptAliveTasks().contains(task2.getTaskId()));
    assertFalse(myManager.getKeptAliveTasks().contains(task3.getTaskId()));
    assertTrue(myManager.getKeptAliveTasks().contains(task4.getTaskId()));
  }

  /**
   * tests delayed tasks for hashcode and equals.
   */
  public void testDelayedTaskhashCodeAndEquals() {
    final Task task1 = new Task("1", WORKER_NAME);
    final Task task2 = new Task("2", WORKER_NAME);
    final DelayedTask dt1 = new DelayedTask(task1, 1, TimeUnit.NANOSECONDS);
    final DelayedTask dt2 = new DelayedTask(task1, 1000, TimeUnit.SECONDS);
    final DelayedTask dt3 = new DelayedTask(task2, 1000, TimeUnit.SECONDS);

    assertEquals(dt1, dt2);
    assertEquals(dt2, dt1);
    assertFalse(dt2.equals(dt3));
    assertFalse(dt1.equals(dt3));
    assertFalse(dt3.equals(dt2));
    assertFalse(dt3.equals(dt1));
    // different objects, Task cannot be equals to DeleayedTask
    assertFalse(dt1.equals(task1));
    assertFalse(task1.equals(dt1));
    assertEquals(dt1.hashCode(), dt2.hashCode());
    assertFalse(dt1.hashCode() == dt3.hashCode());
    assertFalse(dt2.hashCode() == dt3.hashCode());
  }

  /**
   * tests if removal of task that is not kept alive does not result in an exception.
   * 
   * @throws InterruptedException
   *           test failed
   */
  public void testTaskKeepAliveRemoveTaskTwice() throws InterruptedException {
    // -- init
    final DummyTaskManager myManager = new DummyTaskManager();
    final TaskKeepAlive taskKeepAlive = new TaskKeepAlive(myManager, 2);
    final Task task1 = new Task("1", WORKER_NAME);
    myManager.addTask(task1);
    taskKeepAlive.addTask(task1);
    taskKeepAlive.removeTask(task1);
    // remove it a second time
    taskKeepAlive.removeTask(task1);
  }

  /**
   * tests if a task for a nonexisting worker does not lead to error.
   * 
   * @throws InterruptedException
   *           test failed
   */
  public void testTaskKeepAliveRemoveTaskForWrongWorker() throws InterruptedException {
    // -- init
    final DummyTaskManager myManager = new DummyTaskManager();
    final TaskKeepAlive taskKeepAlive = new TaskKeepAlive(myManager, 1);
    taskKeepAlive.addKeepAliveFailureListener(myManager);
    final long scheduleTime = 1000;
    _scheduledExecutor.scheduleAtFixedRate(taskKeepAlive, scheduleTime, scheduleTime, TimeUnit.MILLISECONDS);
    final Task task1 = new Task("1", "iDoNotExist");
    myManager.addTask(task1);
    taskKeepAlive.addTask(task1);

    Thread.sleep(scheduleTime);
    Thread.sleep(scheduleTime);
    Thread.sleep(scheduleTime);
    Thread.sleep(scheduleTime);

    // schould be enough
    // must be kept alive only once...
    assertEquals(1, myManager._keptAliveTasks.size());
    taskKeepAlive.removeKeepAliveFailureListener(myManager);

    // myManager must have been informed about the removal of the task.
    assertTrue(myManager.getRemovedTasks().contains(task1));
    assertEquals(1, myManager.getRemovedTasks().size());
  }
}
