/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.admin;

import java.util.List;

/**
 * The SolrAdmin interface.
 * 
 * @author pwissel
 * 
 */
public interface SolrAdmin {

  /**
   * Action alias. (Experimental) Adds an additional name for a core. The example below allows access to the same core
   * via the names "core0" and "corefoo".
   * 
   * http://localhost:8983/solr/admin/cores?action=ALIAS&core=core0&other=corefoo
   * 
   * @return the response message.
   */
  String alias(final String coreName, final String aliasName) throws Exception;

  /**
   * Action create. Creates a new core based on preexisting instanceDir/solrconfig.xml/schema.xml, and registers it. If
   * persistence is enabled (persist=true), the configuration for this new core will be saved in 'solr.xml'. If a core
   * with the same name exists, while the "new" created core is initalizing, the "old" one will continue to accept
   * requests. Once it has finished, all new request will go to the "new" core, and the "old" core will be unloaded.
   * 
   * http://localhost:8983/solr/admin/cores?action=CREATE&name=coreX&instanceDir=path_to_instance_directory&config=
   * config_file_name.xml&schema=schem_file_name.xml&dataDir=data
   * 
   * instanceDir is a required parameter. config, schema & dataDir parameters are optional. (Default is to look for
   * solrconfig.xml/schema.xml inside instanceDir. Default place to look for dataDir depends on solrconfig.xml.)
   * 
   * @param coreName
   *          the core name.
   * @param instanceDir
   *          the instance directory.
   * @param dataDir
   *          the data directory.
   * @return the response message.
   */
  String create(final String coreName, final String instanceDir, final String dataDir) throws Exception;

  /**
   * copies the definition of the source core and creates the new target core for it in the default location.
   * <p/>
   * Only supported in embedded mode.
   * 
   * @return the response message.
   */
  String create(String srcCore, String targetCoreName) throws Exception;

  /**
   * Action merge. CoreAdminHandler now supports merging one or more indexes into another index (since Solr 1.4).
   * 
   * http://localhost:8983/solr/admin/cores?action=mergeindexes&core=core0&indexDir=/opt/solr/core1/data/index&indexDir=
   * /opt/solr/core2/data/index
   * 
   * The above command will merge the indexes of core1 and core2 into core0. The path for this command is the
   * 'adminPath' defined in solr.xml (default is /admin/cores).
   * 
   * Before executing this command, one must make sure to call commit on core1 and core2 (in order to close IndexWriter)
   * and no writes should happen on core1 and core2 until the merge command completes. Failure to do so may corrupt the
   * core0 index. Once the merge is completed, a commit should be called on core0 to make the changes visible to
   * searchers.
   * 
   * NOTE: In this example core0 must exist and have a compatible schema with core1 and core2. The 'mergeindexes'
   * command will not create a new core named 'core0' if it does not already exist.
   * 
   * @param coreName
   *          the target core name.
   * @param indexDir1
   *          the first source index directory.
   * @param indexDir2
   *          the second source index directory.
   * @return the response message.
   */
  String merge(final String coreName, final String indexDir1, final String indexDir2) throws Exception;

  /**
   * Action reload. Load a new core from the same configuration as an existing registered core. While the "new" core is
   * initalizing, the "old" one will continue to accept requests. Once it has finished, all new request will go to the
   * "new" core, and the "old" core will be unloaded.
   * 
   * http://localhost:8983/solr/admin/cores?action=RELOAD&core=core0
   * 
   * This can be useful when (backwards compatible) changes have been made to your solrconfig.xml or schema.xml files
   * (ie: new <field> declarations, changed default params for a <requestHandler>, etc...) and you want to start using
   * them without stopping and restarting your whole Servlet Container.
   * 
   * @return the response message.
   */
  String reload(final String coreName) throws Exception;

  /**
   * Action rename. Change the names used to access a core. The example below changes the name of the core from "core0"
   * to "core5".
   * 
   * http://localhost:8983/solr/admin/cores?action=RENAME&core=core0&other=core5
   * 
   * @return the response message.
   */
  String rename(final String coreName, final String newName) throws Exception;

  /**
   * Action status. Get the status for a given core or all cores if no core is specified:
   * 
   * http://localhost:8983/solr/admin/cores?action=STATUS&core=core0
   * 
   * http://localhost:8983/solr/admin/cores?action=STATUS
   * 
   * @return the response message.
   */
  String status(final String coreName) throws Exception;

  /**
   * Removes a core from solr. Existing requests will continue to be processed, but no new requests can be sent to this
   * core by the name. If a core is registered under more than one name, only that specific mapping is removed.
   * 
   * http://localhost:8983/solr/admin/cores?action=UNLOAD&core=core0
   * 
   * @return the response message.
   */
  String unload(final String coreName) throws Exception;

  /** get all names of cores available in this server. */
  List<String> getCoreNames() throws Exception;

  /** get the names of fields defined in the named core. */
  List<String> getFieldNames(final String coreName) throws Exception;

  /** clear all internal caches after external admin operations (e.g. cached field names). */
  void clearCoreCaches();

  /** clear internal caches for named core after external admin operations (e.g. cached field names). */
  void clearCoreCache(String coreName);

}
