/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.internal;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.bulkbuilder.BulkbuilderConstants;
import org.eclipse.smila.bulkbuilder.helper.BulkbuilderBase;
import org.eclipse.smila.bulkbuilder.helper.BulkbuilderTaskProvider;
import org.eclipse.smila.bulkbuilder.helper.MicroBulkbuilder;
import org.eclipse.smila.bulkbuilder.outputs.BulkType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.utils.conversion.DataSizeParser;

/**
 * BulkBuilder implementation.
 * 
 * Creates bulks from the pushed records on a time and size limit.
 * 
 */
public class Bulkbuilder extends BulkbuilderBase {

  /** the bulk builder configuration. */
  private final BulkbuilderConfig _bulkbuilderConfig;

  /** local Logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Creates a new BulkBuilder instance.
   * 
   * @param objectStore
   *          a reference to the objectStore the object store service.
   * @param taskProvider
   *          the task provider that generates tasks for the bulkbuilder
   * @throws BlackboardAccessException
   *           cannot access blackboard
   */
  public Bulkbuilder(final BulkbuilderConfig config, final ObjectStoreService objectStore,
    final BulkbuilderTaskProvider taskProvider) throws BlackboardAccessException {
    super(objectStore, taskProvider, new MicroBulkbuilder(config.getMaxParallelMicroBulks()));
    _bulkbuilderConfig = config;
  }

  /**
   * @return the timeout for committing bulks either from task parameter or from bulkbuilder configuration for bulk
   *         mode.
   */
  @Override
  protected long getCommitTimeoutMillis(final AnyMap taskParameters) {
    long result = -1;
    result = _bulkbuilderConfig.getLimitTimeMillis();
    // task parameter overwrites bulkbuilder configuration
    if (taskParameters.containsKey(BulkbuilderConstants.TASK_PARAM_LIMIT_TIME)) {
      try {
        result =
          taskParameters.getLongValue(BulkbuilderConstants.TASK_PARAM_LIMIT_TIME)
            * BulkbuilderConfig.MILLISECONDS_MULTIPLIER;
      } catch (final NumberFormatException e) {
        _log.warn("Invalid value '" + taskParameters.get(BulkbuilderConstants.TASK_PARAM_LIMIT_TIME)
          + "' for task parameter '" + BulkbuilderConstants.TASK_PARAM_LIMIT_TIME + "'.");
      }
    }
    return result;
  }

  /**
   * @return the maximum bulk size either from task parameter or from bulkbuilder configuration.
   */
  @Override
  protected long getBulkSizeLimit(final BulkType bulkType, final AnyMap taskParameters) {
    long result = 0;
    result = _bulkbuilderConfig.getLimitSize();
    if (taskParameters.containsKey(BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE)) {
      try {
        result = DataSizeParser.parse(taskParameters.getStringValue(BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE));
      } catch (final Exception e) {
        _log.warn("Invalid value '" + taskParameters.get(BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE)
          + "' for task parameter '" + BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE + "'.");
      }
    }
    return result;
  }
}
