/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.zookeeper.internal;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.server.quorum.QuorumPeerConfig;
import org.eclipse.smila.clusterconfig.ClusterConfigException;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;
import org.eclipse.smila.zookeeper.ZooKeeperService;
import org.osgi.service.component.ComponentContext;

/**
 * OSGi services that configures and starts a ZooKeeper, either stand-alone if we are not running in a multi-node
 * cluster or without ClusterConfigService at all or as distributed on the complete cluster.
 */
public class ZooKeeperServiceImpl implements ZooKeeperService, Watcher {

  /** name of configuration file. */
  private static final String CONFIG_FILENAME = "zoo.cfg";

  /** Timeout in seconds for client to wait for reconnection before giving up. */
  private static final int WAIT_FOR_RECONNECTION_TIMEOUT = 60;

  /** multiplicator to create milliseconds from minutes. */
  private static final long MINUTES_TO_MILLISECONDS = 60 * 1000;

  /** name of file containing the ID of the server we start. */
  private static final String MYID_FILE_NAME = "myid";

  /** name of environment variable to change zookeeper data directory for tests. */
  private static final String ENV_DATADIR = "SMILA_ZK_DATADIR";

  /** name of property specifying the snapshot directory in the ZooKeeper props. */
  private static final String PROP_DATADIR = "dataDir";

  /** name of property specifying the transaction log directory in the ZooKeeper props. */
  private static final String PROP_DATALOGDIR = "dataLogDir";

  /** name of the property specifying the ZooKeeper port used by followers to connect to the leader. */
  private static final String PROPERTY_ZK_SERVER_PORT = "zk.serverPort";

  /** name of the property specifying the ZooKeeper port for leader election. */
  private static final String PROPERTY_ZK_ELECTION_PORT = "zk.electionPort";

  /** name of property specifying number of snapshots to keep on disk. */
  private static final String PROP_SNAPSHOTSTOKEEP = "zk.snapshotsToKeep";

  /** default ZooKeeper server port. */
  private static final int DEFAULT_ZK_SERVER_PORT = 2888;

  /** default ZooKeeper election port. */
  private static final int DEFAULT_ZK_ELECTION_PORT = 3888;

  /** default number of snapshots to keep on disk. */
  private static final int DEFAULT_SNAPSHOTSTOKEEP = 5;

  /** minimal number of snapshots to keep on disk. */
  private static final int MINIMUM_SNAPSHOTSTOKEEP = 3;

  /** The IP-Addresses of cluster nodes. */
  private ArrayList<String> _clusterNodeAddresses;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** configuration for cluster peer. */
  private QuorumPeerConfig _quorumPeerConfig;

  /** server watchdog thread. */
  private Thread _serverRunnerThread;

  /** server watchdog. */
  private ZooKeeperServerRunner _serverRunner;

  /** connection to ClusterConfigService. */
  private ClusterConfigService _clusterService;

  /** current failsafety level. */
  private long _maxFailedNodes = -1;

  /** Garbage collector for ZooKeeper data file management. */
  private ZooKeeperGC _gc;

  /** Thread running garbage collector. */
  private Thread _gcThread;

  /**
   * single Zookeeper client instance, administered by this service. From Zookeeper documentation: "The instantiated
   * ZooKeeper client object will pick an arbitrary server from the connectString and attempt to connect to it. If
   * establishment of the connection fails, another server in the connect string will be tried (the order is
   * non-deterministic, as we random shuffle the list), until a connection is established. The client will continue
   * attempts until the session is explicitly closed (or the session is expired by the server)".
   */
  //TODO use volatile? because of doublecheck pattern (see getClient()/createClient()) - but this would be slower...  
  private ZooKeeper _zkClient;

  /** State of zkClient (SyncConnected, Disconnected, ...). */
  private KeeperState _zkClientState;

  /** Lock for handling concurrent processing of zookeeper client state changes. */
  //"When set true, under contention, lock favors granting access to the longest-waiting thread."
  private final Lock _zkEventLock = new ReentrantLock(true);

  /** Condition for await/signalAll mechanism associated to _zkEventLock. */
  private final Condition _zkEventLockCondition = _zkEventLock.newCondition();

  /** Watchers for the Zookeper events. */
  private final Collection<Watcher> _watchers = new CopyOnWriteArraySet<Watcher>();

  /**
   * {@inheritDoc}
   */
  @Override
  public ZooKeeper getClient() throws IOException, ClusterConfigException {
    if (_zkClient == null) {
      final int timeout = _quorumPeerConfig.getMaxSessionTimeout();
      createClient(timeout, this);
    }
    return _zkClient;
  }

  /** create zookeeper client. */
  private void createClient(final int sessionTimeout, final Watcher watcher) throws IOException,
    ClusterConfigException {
    _zkEventLock.lock();
    try {
      if (_zkClient == null) {
        final StringBuffer connectionString = new StringBuffer();
        if (_clusterNodeAddresses != null && _clusterNodeAddresses.size() > 0) {
          for (final String node : _clusterNodeAddresses) {
            if (connectionString.length() > 0) {
              connectionString.append(",");
            }
            connectionString.append(node + ":" + _quorumPeerConfig.getClientPortAddress().getPort());
          }
        } else {
          // doesn't have cluster config available or one node cluster, so use localhost.
          connectionString.append("127.0.0.1" + ":" + _quorumPeerConfig.getClientPortAddress().getPort());
        }
        _log.debug("Connecting to Zookeper with connectionString " + connectionString + ", timeout is "
          + sessionTimeout + " ms, Watcher is instance of " + watcher.getClass());
        _zkClient = new ZooKeeper(connectionString.toString(), sessionTimeout, watcher);
      }
    } finally {
      _zkEventLock.unlock();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void closeClient() {
    if (_zkClient != null) {
      _zkEventLock.lock();
      try {
        if (_zkClient != null) {
          if (_log.isTraceEnabled()) {
            _log.trace("closeClient(): closing client.");
          }
          _zkClient.close();
        }
      } catch (final Exception ex) {
        _log.warn("Error while closing zookeeper client", ex);
      } finally {
        _zkClient = null;
        _zkEventLock.unlock();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getFailSafetyLevel() {
    if (_maxFailedNodes < 0) {
      if (isClusterStart()) {
        _maxFailedNodes = _clusterService.getFailSafetyLevel();
      } else {
        _maxFailedNodes = 0;
      }
    }
    return _maxFailedNodes;
  }

  /**
   * OSGi Declarative Services service activation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void activate(final ComponentContext context) {
    final InputStream propStream = ConfigUtils.getConfigStream(BUNDLE_ID, CONFIG_FILENAME);
    try {
      final Properties props = new Properties();
      props.load(propStream);
      final String dataDir = prepareDataDir(props);
      final String dataLogDir = prepareDataLogDir(props);
      createMyIdFile(props);
      addQuorumPeersToConfig(props);
      _quorumPeerConfig = new QuorumPeerConfig();
      _quorumPeerConfig.parseProperties(props);
      _serverRunner = new ZooKeeperServerRunner(_quorumPeerConfig, isClusterStart(), getMyId());
      _serverRunnerThread = new Thread(_serverRunner, "ZooKeeperServerRunner");
      _serverRunnerThread.start();
      final long gcInterval = getGCInterval();
      if (gcInterval > 0) {
        final int snapshotsToKeep = getSnapshotsToKeep(props);
        final long gcIntervalInMillis = gcInterval * MINUTES_TO_MILLISECONDS;
        _gc = new ZooKeeperGC(gcIntervalInMillis, dataDir, dataLogDir, snapshotsToKeep);
        _gcThread = new Thread(_gc, "ZooKeeperService:ZooKeeperGC");
        _gcThread.start();
        _log.info("GC started with interval " + gcInterval + " minutes and " + snapshotsToKeep + " files to keep.");
      } else {
        _log.info("GC is disabled!");
      }
    } catch (final Throwable ex) {
      final String msg = "Error while activating " + BUNDLE_ID;
      _log.error(msg, ex);
      throw new RuntimeException(msg, ex);
    } finally {
      IOUtils.closeQuietly(propStream);
    }
  }

  /**
   * OSGi Declarative Services service deactivation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void deactivate(final ComponentContext context) {
    _maxFailedNodes = -1;
    if (_gc != null) {
      _gc.stop();
    }
    if (_gcThread != null) {
      try {
        _gcThread.join();
      } catch (final InterruptedException e) {
        _log.error("Interrupted while waiting for GC shutdown", e);
      }
    }
    _gc = null;
    _gcThread = null;
    if (_serverRunnerThread != null && _serverRunnerThread.isAlive()) {
      _serverRunner.shutdown();
      try {
        _serverRunnerThread.join();
      } catch (final InterruptedException e) {
        _log.warn("Interrupted while waiting for server to shutdown", e);
      }
    }
    closeClient();
  }

  /**
   * determine snapshot dir (either defined in properties or in SMILA workspace), ensure that it is empty and adapt
   * properties.
   */
  private String prepareDataDir(final Properties props) {
    String dataDir = props.getProperty(PROP_DATADIR);
    if (dataDir == null) {
      dataDir = System.getenv(ENV_DATADIR);
    }
    boolean dataDirOk = false;
    if (dataDir != null) {
      try {
        ensureEmptyDirectory(new File(dataDir));
        dataDirOk = true;
      } catch (final IOException ex) {
        _log.error("Could not create clean data directory " + dataDir + ", trying directory in workspace.", ex);
      }
    }
    if (!dataDirOk) {
      File workingDir = null;
      try {
        workingDir = WorkspaceHelper.createWorkingDir(BUNDLE_ID);
        ensureEmptyDirectory(workingDir);
      } catch (final IOException ex) {
        _log.error("Could not create clean data directory in workspace for " + BUNDLE_ID
          + ", creating temp directory. Please check workspace directory.", ex);
        try {
          workingDir = File.createTempFile(BUNDLE_ID + "-", "");
          ensureEmptyDirectory(workingDir);
        } catch (final IOException ex2) {
          _log.error("Error creating clean fallback working directory, strange behaviour may occur. Good luck.",
            ex2);
        }
      }
      dataDir = workingDir.getAbsolutePath();
    }
    props.setProperty(PROP_DATADIR, dataDir);
    _log.info("ZooKeeper snapshot data directory is " + dataDir);
    return dataDir;
  }

  /**
   * ensure empty transaction log directory, if specified separately. Otherwise adapt props to use snapshot dir for
   * logs, too.
   */
  private String prepareDataLogDir(final Properties props) {
    final String dataDir = props.getProperty(PROP_DATADIR);
    String dataLogDir = props.getProperty(PROP_DATALOGDIR);
    boolean dataLogDirOk = false;
    if (dataLogDir != null) {
      try {
        ensureEmptyDirectory(new File(dataLogDir));
        dataLogDirOk = true;
      } catch (final IOException ex) {
        _log.error("Could not create clean transaction log directory " + dataDir
          + ", using data directory for transaction logs, too.", ex);
      }
    }
    if (!dataLogDirOk) {
      dataLogDir = dataDir;
      props.put(PROP_DATALOGDIR, dataLogDir);
    }
    _log.info("ZooKeeper transaction log directory is " + dataLogDir);
    return dataLogDir;
  }

  /**
   * get property value for {@link #PROP_SNAPSHOTSTOKEEP}. Use {@link #DEFAULT_SNAPSHOTSTOKEEP}, if not set, or
   * {@link #MINIMUM_SNAPSHOTSTOKEEP} if value to small.
   */
  private int getSnapshotsToKeep(final Properties props) {
    int snapshotsToKeep = getIntProperty(props, PROP_SNAPSHOTSTOKEEP, DEFAULT_SNAPSHOTSTOKEEP);
    if (snapshotsToKeep < MINIMUM_SNAPSHOTSTOKEEP) {
      _log.info("The value of property " + PROP_SNAPSHOTSTOKEEP + " must not be less than "
        + MINIMUM_SNAPSHOTSTOKEEP + ", correcting.");
      snapshotsToKeep = MINIMUM_SNAPSHOTSTOKEEP;
    }
    return snapshotsToKeep;
  }

  /** parse integer property value from props, if set, return defaultValue, if not set or not a valid integer. */
  private int getIntProperty(final Properties props, final String name, final int defaultValue) {
    final String propValue = props.getProperty(name);
    if (propValue != null) {
      try {
        return Integer.parseInt(propValue.trim());
      } catch (final NumberFormatException ex) {
        _log.warn("Invalid value " + propValue + " for property " + name + ", using default value " + defaultValue
          + " instead.", ex);
      }
    }
    return defaultValue;
  }

  /**
   * delete an existing file or directory at the given location and create a new directory instead.
   * 
   * @param location
   *          file to delete.
   * @throws IOException
   *           error deleting file.
   */
  private void ensureEmptyDirectory(final File location) throws IOException {
    if (location.exists()) {
      if (location.isDirectory()) {
        FileUtils.deleteDirectory(location);
      } else {
        if (!location.delete()) {
          throw new IOException("Could not delete " + location);
        }
      }
    }
    if (!location.mkdir()) {
      throw new IOException("Could not create new directory at " + location);
    }
  }

  /**
   * create MyId File.
   * 
   * @param props
   *          configuration properties containing dataDir setting
   * @throws ClusterConfigException
   *           error checking the cluster configuration.
   * @throws IOException
   *           error creating the file.
   */
  private void createMyIdFile(final Properties props) throws IOException, ClusterConfigException {
    final String dataDir = props.getProperty(PROP_DATADIR);
    final File dataDirFile = new File(dataDir, MYID_FILE_NAME);
    final long myid = getMyId();
    BufferedWriter myidOut = null;
    try {
      final FileWriter myidWriter = new FileWriter(dataDirFile); // overwrite myid file if exists
      myidOut = new BufferedWriter(myidWriter);
      myidOut.write(String.valueOf(myid));
      myidOut.flush();
    } finally {
      if (myidOut != null) {
        myidOut.close();
      }
    }
  }

  /**
   * add quorum peer nodes as determined by ClusterConfigService to configuration properties. initialize servers as
   * observers or followers.
   * 
   * 
   * @param props
   *          properties to adapt.
   * @throws ClusterConfigException
   *           error reading cluster config.
   */
  private void addQuorumPeersToConfig(final Properties props) throws ClusterConfigException {
    long nodeNo = 0;
    if (isClusterStart()) { // properties not needed for stand-alone server.
      final long maxFailedNodes = getFailSafetyLevel();
      if (isObserverNode(getMyId(), maxFailedNodes)) {
        props.put("peerType", "observer");
      }

      final int serverPort = getIntProperty(props, PROPERTY_ZK_SERVER_PORT, DEFAULT_ZK_SERVER_PORT);
      final int electionPort = getIntProperty(props, PROPERTY_ZK_ELECTION_PORT, DEFAULT_ZK_ELECTION_PORT);

      findIPAddressesOfClusterNodes(serverPort);

      for (final String node : _clusterService.getClusterNodes()) {
        ++nodeNo;
        final String key = "server." + nodeNo;
        final StringBuffer value = new StringBuffer(node + ":" + serverPort + ":" + electionPort);
        if (isObserverNode(nodeNo, maxFailedNodes)) {
          value.append(":observer");
        }
        props.put(key, value.toString());
      }
    }
  }

  /**
   * Determines if a server in a ZooKeeper server ensemble is an observer.
   * 
   * @param nodeNo
   *          The ZooKeeper server node's order number.
   * @param maxFailedNodes
   *          The maximum number of servers that may fail if the task manager fail safety is to be guaranteed.
   * @return true if the server is an observer, false otherwise.
   */
  private static boolean isObserverNode(final long nodeNo, final long maxFailedNodes) {
    return nodeNo > 0 && maxFailedNodes > 0 && nodeNo > maxFailedNodes * 2 + 1;
  }

  /**
   * get the position of my host in the cluster node list.
   * 
   * @return position in cluster node list, starting with 1
   * @throws ClusterConfigException
   *           error reading cluster config.
   */
  private long getMyId() throws ClusterConfigException {
    long nodeNo = 0;
    if (_clusterService.isConfigured()) {
      final String myHost = _clusterService.getLocalHost();
      for (final String node : _clusterService.getClusterNodes()) {
        nodeNo++;
        if (node.equals(myHost)) {
          return nodeNo;
        }
      }
      throw new ClusterConfigException("Could not find local host " + myHost + " in cluster nodes: "
        + _clusterService.getClusterNodes());
    }
    return nodeNo;
  }

  /**
   * 
   * @return true if we are running in a multi-node cluster
   */
  private boolean isClusterStart() {
    try {
      return _clusterService.isConfigured() && _clusterService.getClusterNodes().size() > 1;
    } catch (final ClusterConfigException e) {
      _log.warn("Error reading cluster config", e);
      return false;
    }
  }

  /**
   * 
   * @return GC interval in minutes as configured in cluster config, if present. Else 0 (deactivated)
   * @throws ClusterConfigException
   *           error accessing cluster config
   */
  private long getGCInterval() throws ClusterConfigException {
    final long gcInterval = _clusterService.getZkGcInterval();
    return gcInterval;
  }

  /**
   * set new ClusterConfigService. To be called by DS runtime before activation.
   * 
   * @param clusterConfigService
   *          new ClusterConfigService
   */
  public void setClusterConfigService(final ClusterConfigService clusterConfigService) {
    _clusterService = clusterConfigService;
  }

  /**
   * remove a ClusterConfigService. To be called by DS runtime after deactivation.
   */
  public void unsetClusterConfigService(final ClusterConfigService clusterConfigService) {
    if (_clusterService == clusterConfigService) {
      _clusterService = null;
    }
  }

  /**
   * Determines the IP addresses of cluster nodes. If an IP address is unresolved, the host name is used instead.
   * 
   * @param port
   *          the Zookeeper port
   */
  private void findIPAddressesOfClusterNodes(final int port) {
    _clusterNodeAddresses = new ArrayList<String>();
    try {
      if (_clusterService.isConfigured()) {
        final List<String> nodes = _clusterService.getClusterNodes();
        final Iterator<String> it = nodes.iterator();
        while (it.hasNext()) {
          final String host = it.next();
          final InetSocketAddress address = new InetSocketAddress(host, port);
          if (!address.isUnresolved()) {
            _clusterNodeAddresses.add(address.getAddress().getHostAddress());
          } else {
            _log.warn("Getting IP sddress of host " + host + " failed. Host name will be used.");
            _clusterNodeAddresses.add(host);
          }
        }
      }
    } catch (final Exception ex) {
      throw new RuntimeException("Getting ip addresses of cluster nodes failed", ex);
    }
  }

  @Override
  public void waitForClientConnected() {
    _zkEventLock.lock(); // must get the lock to call await on condition (await releases the lock!)
    try {
      boolean stillWaiting = true;
      boolean triedClientRecreation = false;
      while (_zkClientState != KeeperState.SyncConnected) {
        if (stillWaiting) {
          stillWaiting = _zkEventLockCondition.await(WAIT_FOR_RECONNECTION_TIMEOUT, TimeUnit.SECONDS);
        } else {
          if (triedClientRecreation) {
            // we already tried to recreate the zookeeper client - didn't help
            throw new RuntimeException("Waited " + WAIT_FOR_RECONNECTION_TIMEOUT
              + " seconds after Zookeeper client recreation");
          } else {
            // client didn't reconnect - try to create a new zookeeper client
            try {
              _log.warn("Waited " + WAIT_FOR_RECONNECTION_TIMEOUT
                + " seconds for Zookeeper client reconnection, will try recreation");
              recreateZkClient();
              triedClientRecreation = true;
              stillWaiting = true;
            } catch (final Exception e) {
              throw new RuntimeException("Error while recreating zookeeper client", e);
            }
          }
        }
      }
    } catch (final InterruptedException e) {
      throw new RuntimeException(e);
    } finally {
      _zkEventLock.unlock();
    }
  }

  /**
   * we only use Watches for zookeeper client state change events.
   */
  @Override
  public void process(final WatchedEvent event) {
    if (_log.isDebugEnabled()) {
      _log.debug("process watched event: " + event.getState());
    }
    try {
      _zkEventLock.lock();

      // handle state change events (normally we don't get no others)      
      if (event.getPath() == null) {
        if (_log.isInfoEnabled()) {
          _log.info("Zookeeper client state changed from '" + _zkClientState + "' to '" + event.getState() + "'");
        }
        _zkClientState = event.getState();

        // zookeeper documentation: 
        // "The ZK client library will handle reconnect for you"
        // "Only create a new session when you are notified of session expiration"    
        if (event.getState() == KeeperState.Expired) {
          _log.warn("got zookeeper session expired event - recreating zookeeper client");
          recreateZkClient();
        }

        // awaken all threads waiting for state changes
        _zkEventLockCondition.signalAll();

      } else {
        // signal explicitly registered watchers. Ignore any errors these watchers may throw.
        for (final Watcher watcher : _watchers) {
          try {
            watcher.process(event);
          } catch (final Throwable t) {
            _log.error("Watcher failed. ", t);
          }
        }
      }

    } catch (final Exception e) {
      throw new RuntimeException("Error while processing zookeeper watched event: " + event, e);
    } finally {
      _zkEventLock.unlock();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void registerWatcher(final Watcher watcher) {
    _watchers.add(watcher);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unregisterWatcher(final Watcher watcher) {
    _watchers.remove(watcher);
  }

  /**
   * Create new zookeeper client (session) to reconnect to zookeeper cluster.
   */
  private void recreateZkClient() throws IOException, ClusterConfigException {
    _log.info("recreating zookeeper client");
    _zkEventLock.lock();
    try {
      closeClient();
      getClient(); // will create new client cause we closed the old one before
    } finally {
      _zkEventLock.unlock();
    }
  }

  @Override
  public AnyMap getServerState() {
    final AnyMap zkState = DataFactory.DEFAULT.createAnyMap();
    final AnyMap localState = DataFactory.DEFAULT.createAnyMap();    
    final String myState = _serverRunner.getServerState();
    localState.put(_clusterService.getLocalHost(), myState);    
    localState.putAll(_serverRunner.getServerStatistics());
    zkState.put("local server", localState);
    if (_clusterService.isConfigured()) {
      try {
        final AnyMap clusterState = DataFactory.DEFAULT.createAnyMap();
        final List<String> nodes = _clusterService.getClusterNodes();
        if (nodes.size() > 1) {
          final long maxFailNodes = _clusterService.getFailSafetyLevel();
          final String leaderConn = _serverRunner.getRemoteLeaderConnection();
          final long noOfFollowers = maxFailNodes * 2 + 1;
          for (int i = 0; i < nodes.size(); i++) {
            final String node = nodes.get(i);
            if (i < noOfFollowers) {
              if (leaderConn != null && leaderConn.contains(node + "/")) {
                // remote leader
                clusterState.put(node, "Leader");
              } else if (node.equals(_clusterService.getLocalHost()) && "leading".equals(myState)) {
                // I am leader
                clusterState.put(node, "Leader");
              } else {
                clusterState.put(node, "Follower");
              }
            } else {
              clusterState.put(node, "Observer");
            }
          }
          zkState.put("cluster", clusterState);
        }
      } catch (final ClusterConfigException e) {
        ; // ignore, it's just for inofficial monitoring
      }
    }
    return zkState;
  }

}
