/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation 
 *               Sebastian Voigt (brox IT Solutions GmbH)
 *               Andreas Weber (Attensity Europe GmbH) - changes for job management
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.impl;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.connectivity.ConnectivityException;
import org.eclipse.smila.connectivity.framework.CrawlState;
import org.eclipse.smila.connectivity.framework.Crawler;
import org.eclipse.smila.connectivity.framework.CrawlerController;
import org.eclipse.smila.connectivity.framework.performancecounters.ConnectivityPerformanceAgent;
import org.eclipse.smila.connectivity.framework.performancecounters.CrawlerPerformanceCounterHelper;
import org.eclipse.smila.connectivity.framework.schema.config.DataConnectionID.DataConnectionType;
import org.eclipse.smila.connectivity.framework.schema.config.DataSourceConnectionConfig;
import org.eclipse.smila.connectivity.framework.util.CrawlThreadState;
import org.eclipse.smila.connectivity.framework.util.CrawlerControllerCallback;
import org.osgi.service.component.ComponentContext;

/**
 * Basic Implementation of a CrawlerController.
 */
public class CrawlerControllerImpl extends AbstractController implements CrawlerController,
  CrawlerControllerCallback {

  /** The Constant BUNDLE_ID. */
  private static final String BUNDLE_ID = "org.eclipse.smila.connectivity.framework";

  /** Maximum time in milliseconds to wait for a CrawlThread to join during deactivation. */
  private static final long CRAWL_THREAD_JOIN_TIME = 5000;

  /** The LOG. */
  private final Log _log = LogFactory.getLog(CrawlerControllerImpl.class);

  /** A Map of active CrawlThreads. (key: dataSourceId) */
  private final Map<String, CrawlThread> _crawlThreads;

  /** A Map of CrawlStates. (key: dataSourceId) */
  private final Map<String, CrawlState> _crawlStates;

  /** A Map of active CrawlThreads. (key: dataSourceId) to store performance counters for finished crawl threads */
  private final Map<String, CrawlerPerformanceCounterHelper<? extends ConnectivityPerformanceAgent>> _crawlCounters;

  /**
   * Default Constructor.
   */
  public CrawlerControllerImpl() {
    if (_log.isTraceEnabled()) {
      _log.trace("Creating CrawlerControllerImpl");
    }
    _crawlThreads = new HashMap<String, CrawlThread>();
    _crawlStates = new HashMap<String, CrawlState>();
    _crawlCounters = new HashMap<String, CrawlerPerformanceCounterHelper<? extends ConnectivityPerformanceAgent>>();
  }

  /**
   * @return CrawlerPerformanceCounterHelper for given data source id.
   */
  @Override
  public CrawlerPerformanceCounterHelper<? extends ConnectivityPerformanceAgent> getCrawlerCounterHelper(
    final String dataSourceId) {
    final CrawlThread crawlThread = _crawlThreads.get(dataSourceId);
    if (crawlThread != null) {
      return crawlThread.getCrawlerCounterHelper();
    }
    return _crawlCounters.get(dataSourceId);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerController#startCrawl()
   */
  @Override
  public int startCrawl(final String dataSourceId, final String jobName) throws ConnectivityException {
    // check parameters
    assertNotEmpty(dataSourceId, "dataSourceId");
    assertNotEmpty(jobName, "jobName");

    // check if data source is already crawled
    if (_crawlThreads.containsKey(dataSourceId)) {
      throw new ConnectivityException("Can't start a new crawl for DataSourceId '" + dataSourceId
        + "'. It is already crawled by another process.");
    }

    try {
      final DataSourceConnectionConfig configuration = getConfiguration(BUNDLE_ID, dataSourceId);
      getConnectivityManager().checkJobIsActive(jobName);
      final Crawler crawler = createInstance(Crawler.class, configuration.getDataConnectionID().getId());
      final int importRunId = crawler.hashCode();

      // initialize the CrawlState
      final CrawlState crawlState = new CrawlState();
      crawlState.setDataSourceId(dataSourceId);
      crawlState.setState(CrawlThreadState.Running);
      crawlState.setStartTime(System.currentTimeMillis());
      crawlState.setImportRunId(Integer.toString(importRunId));
      _crawlStates.put(dataSourceId, crawlState);

      // initialize the CrawlThread
      final CrawlThread crawlThread =
        new CrawlThread(this, crawlState, getConnectivityManager(), getDeltaIndexingManager(),
          getCompoundManager(), crawler, configuration, jobName);
      _crawlThreads.put(dataSourceId, crawlThread);

      // start CrawlThread
      crawlThread.start();
      return importRunId;
    } catch (final ConnectivityException e) {
      throw e;
    } catch (final Exception e) {
      final String msg = "Error during executeCrawl of DataSourceId '" + dataSourceId + "'";
      if (_log.isErrorEnabled()) {
        _log.error(msg, e);
      }
      throw new ConnectivityException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerController#stopCrawl(java.lang.String)
   */
  @Override
  public void stopCrawl(final String dataSourceId) throws ConnectivityException {
    // check parameters
    assertNotEmpty(dataSourceId, "dataSourceId");

    final CrawlThread crawlThread = _crawlThreads.get(dataSourceId);
    if (crawlThread == null) {
      final String msg = "Could not stop crawl for DataSourceId '" + dataSourceId + "'. No CrawlThread exists.";
      if (_log.isErrorEnabled()) {
        _log.error(msg);
      }
      throw new ConnectivityException(msg);
    }

    crawlThread.stopCrawl();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerController#hasActiveCrawls()
   */
  @Override
  public boolean hasActiveCrawls() throws ConnectivityException {
    return !_crawlThreads.isEmpty();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerController#getAgentTasksState()
   */
  @Override
  public Map<String, CrawlState> getCrawlerTasksState() {
    final HashMap<String, CrawlState> states = new HashMap<String, CrawlState>();
    states.putAll(_crawlStates);
    return states;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.util.CrawlerControllerCallback#unregister(java.lang.String)
   */
  @Override
  public void unregister(final String dataSourceId) {
    final CrawlThread crawlThread = _crawlThreads.get(dataSourceId);
    if (crawlThread != null) {
      _crawlCounters.put(dataSourceId, crawlThread.getCrawlerCounterHelper());
    }
    _crawlThreads.remove(dataSourceId);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerController#getAvailableCrawlers()
   */
  @Override
  public Collection<String> getAvailableCrawlers() {
    return getAvailableFactories();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerController#getAvailableConfigurations()
   */
  @Override
  public Collection<String> getAvailableConfigurations() {
    return getConfigurations(BUNDLE_ID, DataConnectionType.CRAWLER);
  }

  /**
   * DS deactivate method.
   * 
   * @param context
   *          the ComponentContext
   * 
   * @throws Exception
   *           if any error occurs
   */
  protected void deactivate(final ComponentContext context) throws Exception {
    if (_log.isInfoEnabled()) {
      _log.info("Deactivating CrawlerController");
    }
    _lock.writeLock().lock();
    try {
      // stop CrawlThreads
      Iterator<Map.Entry<String, CrawlThread>> it = _crawlThreads.entrySet().iterator();
      while (it.hasNext()) {
        final Map.Entry<String, CrawlThread> entry = it.next();
        try {
          if (entry.getValue() != null) {
            entry.getValue().stopCrawl();
          }
        } catch (final Exception e) {
          if (_log.isErrorEnabled()) {
            _log.error("Error stopping CrawlThread for data source " + entry.getKey(), e);
          }
        }
      }
      // stop CrawlThreads
      it = _crawlThreads.entrySet().iterator();
      while (it.hasNext()) {
        final Map.Entry<String, CrawlThread> entry = it.next();
        try {
          if (entry.getValue() != null) {
            entry.getValue().join(CRAWL_THREAD_JOIN_TIME);
          }
        } catch (final Exception e) {
          if (_log.isErrorEnabled()) {
            _log.error("Error joining CrawlThread for data source " + entry.getKey(), e);
          }
        }
      }
    } finally {
      // Thread.sleep(5000);
      _lock.writeLock().unlock();
    }
  }
}
