/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.AnyMap;

/**
 * Definition of a (persistent) bucket. JSON format:
 * 
 * <pre>
 *      {
 *         "name": "docsBucket",
 *         "type": "recordBulks",
 *         "parameters": 
 *         {
 *            "param1": "value1",
 *            ...
 *            "paramN": "valueN"
 *         }
 *      }
 * </pre>
 */
public class BucketDefinition extends DefinitionBase {

  /** name of JSON property for lists of buckets. */
  public static final String KEY_BUCKETS = "buckets";

  /** name of JSON property for bucket type. */
  public static final String KEY_TYPE = "type";

  /** name of JSON property for bucket modes. */
  public static final String KEY_MODE = "mode";

  /** name of JSON property for bucket parameters. */
  public static final String KEY_PARAMETERS = "parameters";

  /** . */
  private final String _dataObjectType;

  /** parameters are optional and describe a list of parameters to be declared for this workflow. */
  private final AnyMap _parameters;

  /**
   * parse bucket definition from Any object.
   * 
   * @param bucket
   *          see class comment for format of Any object
   * @throws InvalidDefinitionException
   *           error parsing object.
   */
  public BucketDefinition(final AnyMap bucket) throws InvalidDefinitionException {
    super(bucket);
    _dataObjectType = AccessAny.getStringRequired(bucket, KEY_TYPE);
    _parameters = bucket.getMap(KEY_PARAMETERS);
  }

  /**
   * create a bucket definition for a transient bucket.
   * 
   * @param name
   *          bucket name
   * @param dataObjectType
   *          data object type name.
   */
  public BucketDefinition(final String name, final String dataObjectType) {
    super(name);
    _dataObjectType = dataObjectType;
    _parameters = null;
  }

  /**
   * Parse a single bucket from an Any object containing a bucket description.
   * 
   * @param bucketAny
   *          bucket as Any.
   * @return parsed bucket definition.
   * @throws InvalidDefinitionException
   *           error parsing Any.
   */
  public static BucketDefinition parseBucket(final AnyMap bucketAny) throws InvalidDefinitionException {
    BucketDefinition workflow = null;
    try {
      workflow = new BucketDefinition(bucketAny);
    } catch (final InvalidDefinitionException ex) {
      throw ex;
    } catch (final Exception ex) {
      throw new InvalidDefinitionException("Invalid any structure", ex);
    }
    return workflow;
  }

  /**
   * @return name of data object type.
   */
  public String getDataObjectType() {
    return _dataObjectType;
  }

  /**
   * @return the parameters
   */
  public AnyMap getParameters() {
    return _parameters;
  }

  /** {@inheritDoc}. */
  @Override
  public AnyMap toAny(final boolean includingAdditionalAttributes) {
    try {
      final AnyMap bucketAny = super.toAny(includingAdditionalAttributes);
      if (!includingAdditionalAttributes) {
        bucketAny.put(KEY_TYPE, _dataObjectType);
        if (_parameters != null) {
          bucketAny.put(KEY_PARAMETERS, _parameters);
        }
      }
      return bucketAny;
    } catch (final Exception ex) {
      throw new IllegalStateException("Failed to create Any object for bucket " + _name, ex);
    }
  }
}
