/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.smila.common.definitions.ValueExpression;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;

/**
 * Helper class for transforming valueExpressions.
 */
public final class ExpressionUtil {

  /**
   * Private constructor to avoid instance creation.
   */
  private ExpressionUtil() {
  }

  /**
   * Transforms a map with value expressions into a map with strings as value.
   * 
   * @param map
   *          The to be transformed map
   * @return The transformed map
   */
  public static AnyMap evaluateParameters(final Map<String, ValueExpression> map) {
    final AnyMap evaluatedMap = DataFactory.DEFAULT.createAnyMap();
    for (final Map.Entry<String, ValueExpression> entry : map.entrySet()) {
      evaluatedMap.put(entry.getKey(), entry.getValue().evaluateExpr(map).getExpression());
    }
    return evaluatedMap;
  }

  /**
   * Transforms a map with value expressions into a map with strings as value.
   * 
   * @param map
   *          The to be transformed map
   * @return The transformed map
   */
  public static AnyMap evaluateParameters(final AnyMap map) {
    if (map == null) {
      return map;
    }
    final AnyMap topLevelValues = DataFactory.DEFAULT.createAnyMap();
    // evaluate top level entries only
    for (final Entry<String, Any> entry : map.entrySet()) {
      if (entry.getValue().isValue()) {
        topLevelValues.put(entry.getKey(), entry.getValue().asValue());
      }
    }
    return (AnyMap) evaluateParameter(map, topLevelValues);
  }

  /**
   * Transforms a map with value expressions into a map with strings as value.
   * 
   * @param param
   *          The param to be transformed
   * @return The transformed param
   */
  public static Any evaluateParameter(final Any param, final AnyMap topLevelValues) {
    if (param.isString()) {
      return DataFactory.DEFAULT.createStringValue(new ValueExpression(param.asValue().asString()).evaluate(
        topLevelValues).toString());
    } else if (param.isMap()) {
      final AnyMap evaluatedMap = DataFactory.DEFAULT.createAnyMap();
      for (final Entry<String, Any> entry : param.asMap().entrySet()) {
        evaluatedMap.put(entry.getKey(), evaluateParameter(entry.getValue(), topLevelValues));
      }
      return evaluatedMap;
    } else if (param.isSeq()) {
      final AnySeq evaluatedSeq = DataFactory.DEFAULT.createAnySeq();
      for (final Any entry : param.asSeq()) {
        evaluatedSeq.add(evaluateParameter(entry, topLevelValues));
      }
      return evaluatedSeq;
    }
    return param;
  }
}
