/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.taskgenerator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.Task;

/**
 * Default Task Generator implementation.
 */
public class DefaultTaskGenerator extends TaskGeneratorBase {

  /**
   * default behavior: with changed input, create one follow up task per changed inputs for each follow up task: create
   * one output bulk per output slot/bucket of the worker/action. without changed input (initial task creation) create
   * one initial task.
   * 
   * {@inheritDoc}
   */
  @Override
  public List<Task> createTasks(final Map<String, List<BulkInfo>> changedInput,
    final Map<String, Bucket> inputBuckets, final Map<String, Bucket> outputBuckets, final AnyMap parameters,
    final String workerName) throws TaskGeneratorException {
    if (changedInput.isEmpty()) {
      return Collections.singletonList(createInitialTask(outputBuckets, parameters, workerName));
    } else {
      final List<Task> tasks = new ArrayList<Task>(changedInput.size());
      for (final Map.Entry<String, List<BulkInfo>> inputSlot : changedInput.entrySet()) {
        final String inputSlotName = inputSlot.getKey();
        final List<BulkInfo> changedBulks = inputSlot.getValue();
        final Bucket inputBucket = inputBuckets.get(inputSlotName);
        for (final BulkInfo changedBulk : changedBulks) {
          final String objectId = changedBulk.getObjectName();
          tasks.add(createSingleBulkTask(objectId, inputSlotName, inputBucket, outputBuckets, parameters,
            workerName));
        }
      }
      return tasks;
    }
  }

  /**
   * {@inheritDoc}
   * 
   * <p>
   * <b>Note:</b> The method will create one task per object in the input bucket.
   * 
   * If there is more than one input bucket defined for the worker, a {@link TaskGeneratorException} will be thrown.
   * </p>
   */
  @Override
  public List<Task> createRunOnceTasks(final Map<String, Bucket> inputBuckets,
    final Map<String, Bucket> outputBuckets, final AnyMap parameters, final String workerName)
    throws TaskGeneratorException {
    if (inputBuckets.size() != 1) {
      throw new TaskGeneratorException("Cannot create tasks for mode '" + JobRunMode.RUNONCE.name()
        + "' and worker '" + workerName + "'. The worker has not exactly one input bucket ('" + inputBuckets
        + "').");
    }
    try {
      return doCreateRunOnceTasks(inputBuckets, outputBuckets, parameters, workerName);
    } catch (final Exception e) {
      throw new TaskGeneratorException("Cannot create tasks for mode '" + JobRunMode.RUNONCE.name()
        + "' and worker '" + workerName + "'.", e);
    }
  }

  /**
   * Creates a list of tasks for given input buckets. One task will be generated for each matching object found in the
   * store of the bucket. The object's name must begin with the bucket's object prefix.
   * 
   * @param inputBuckets
   *          key: input slot name, value: bucket for this slot
   * @param outputBuckets
   *          key: output slot name, value: bucket for this slot
   * @param parameters
   *          key: param name, value: (evaluated) param value
   * @param workerName
   *          (action) worker for which to create new tasks
   * @return a list of new generated tasks
   * @throws org.eclipse.smila.objectstore.ObjectStoreException
   *           error reading object infos from store.
   */
  protected List<Task> doCreateRunOnceTasks(final Map<String, Bucket> inputBuckets,
    final Map<String, Bucket> outputBuckets, final AnyMap parameters, final String workerName) throws Exception {
    final List<Task> taskList = new ArrayList<Task>();
    for (final Entry<String, Bucket> inputBucketEntry : inputBuckets.entrySet()) {
      final String slotName = inputBucketEntry.getKey();
      final Bucket bucket = inputBucketEntry.getValue();
      for (final StoreObject objectInfo : ObjectStoreRetryUtil.retryGetStoreObjectInfos(_objectStore,
        bucket.getStoreName(), bucket.getDataObjectNamePrefix())) {
        // ignore empty objects.
        if (objectInfo.getSize() > 0) {
          final String objectId = objectInfo.getId();
          taskList.add(createSingleBulkTask(objectId, slotName, bucket, outputBuckets, parameters, workerName));
        }
      }
    }
    return taskList;
  }

  /**
   * @return an initial task
   */
  protected Task createInitialTask(final Map<String, Bucket> outputBuckets, final AnyMap parameters,
    final String workerName) {
    final Task task = createTask(workerName, parameters);
    addOutputBulks(task, null, outputBuckets, parameters);
    return task;
  }
}
