/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore;

import java.io.InputStream;
import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;

/**
 * An ObjectStoreService is mainly used in SMILA to store binary data during bulk processing. Data objects in this
 * service are for example big bulks of records that are to be processed in a single step by some worker. Of course, the
 * ObjectStoreService can also be used to store small objects like record attachments or other auxiliary data. Apart
 * from reading and writing objects (either as blobs or using streams), blobs can be appended to objects, objects can be
 * removed, and it is possible to get a list of descriptions ({@link StoreObject}) about the current objects in a store,
 * optionally filtered by a prefix string on the object ID.
 * 
 * Objects are identified by a {@link String} identifiers. All implementations must support these chars::
 * <ul>
 * <li>ASCII characters: 'a'-'z', 'A'-'Z'
 * <li>numbers: '0'-'9'
 * <li>underscores, dashes and slashes: '_', '-', '/'
 * </ul>
 * Specific implementations may allow more characters.
 * 
 * The ObjectStoreService manages objects in stores. A store can be thought of as a container of objects of the same
 * kind. A store must be created before it can be used to store and fetch objects. In some implementations of this
 * service it is possible that different stores have different properties, for example concerning the reliability of
 * this store like replication of the data objects. Such properties must be specified at creation time, or the service
 * will create a store with default properties. The properties that can be used are dependant on the specific service
 * implementation.
 */
public interface ObjectStoreService {

  /** key of store name of the store info. */
  String KEY_STORE_NAME = "storeName";

  /** key of store properties of the store info. */
  String KEY_STORE_PROPERTIES = "storeProperties";

  /** key of store object count of the store info. */
  String KEY_OBJECT_COUNT = "objectCount";

  /** key of store size of the store info. */
  String KEY_SIZE = "size";

  /** key of store objects of the store info. */
  String KEY_OBJECTS = "objects";

  // store handling methods

  /**
   * @return the names of all currently existing stores.
   * @throws ObjectStoreException
   *           on error.
   */
  Collection<String> getStoreNames() throws ObjectStoreException;

  /**
   * Creates a store with the given configuration properties. If properties are <code>null</code>, the service chooses
   * default properties on its own.
   * 
   * @param storeName
   *          name of store to create
   * @param storeProperties
   *          store configuration, may be null.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws StoreExistsException
   *           if a store with this name exists already.
   * @throws ObjectStoreException
   *           other errors.
   */
  void createStore(String storeName, AnyMap storeProperties) throws ObjectStoreException;

  /**
   * Ensures that a store with the given name existing, regardless of the exact configuration properties. If the store
   * does not exist yet, it is created with some default properties.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws ObjectStoreException
   *           other errors
   */
  void ensureStore(String storeName) throws ObjectStoreException;

  /**
   * Check if the argument is a valid store name for this service implementation.
   * 
   * @return true if the string can be used as a store name, else false.
   */
  boolean isValidStoreName(String storeName);

  /**
   * @return true if a store with this name exists, else false.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws ObjectStoreException
   *           some error.
   */
  boolean existsStore(String storeName) throws ObjectStoreException;

  /**
   * remove a store completely, if it exists. All objects in this store will be lost. No exception is thrown if the
   * store does not exists currently.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws ObjectStoreException
   *           error deleting the store.
   */
  void removeStore(String storeName) throws ObjectStoreException;

  /**
   * Get information about all current objects in the named store.
   * 
   * @return list of {@link StoreObject}s
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error determining the object info list.
   */
  Collection<StoreObject> getStoreObjectInfos(String storeName) throws ObjectStoreException;

  /**
   * Get information about all current objects in the named store that have an ID that starts with the given prefix. If
   * recursive is true all objects within sub directories are returned, otherwise only the top level elements of the
   * store.
   * 
   * @return list of {@link StoreObject}s with IDs matching the prefix.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error determining the object info list.
   */
  Collection<StoreObject> getStoreObjectInfos(String storeName, String objectIdPrefix) throws ObjectStoreException;

  /**
   * Count all current objects in the named store that have an ID that starts with the given prefix. If
   * recursive is true all objects within sub directories are returned, otherwise only the top level elements of the
   * store.
   * 
   * @return list of {@link StoreObject}s with IDs matching the prefix.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error determining the object info list.
   */
  long countStoreObjects(String storeName, String objectIdPrefix) throws ObjectStoreException;
  
  /**
   * Get all prefixes of objectIds (this can also include complete ObjectIds) in the named store that have an ID that
   * starts with the given prefix until the next separator (/). The returned list contains the full qualified objectIds.
   * 
   * @return list of {@link String}s with IDs matching the prefix.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error determining the object info list.
   */
  Collection<String> getPrefixes(String storeName, String objectIdPrefix) throws ObjectStoreException;

  /**
   * get configuration properties of the named store.
   * 
   * @return configuration properties.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws ObjectStoreException
   *           error reading store properties.
   */
  AnyMap getStoreProperties(String storeName) throws ObjectStoreException;

  /**
   * get a description of the current store state to be displayed in administration tools (e.g. by an Http handler). It
   * should include the store properties and the complete object list, if <code>includeObjectInfos == true</code>. It
   * can contain additional information.
   * 
   * @param includeObjectInfos
   *          set to true to include the list of current object infos in the result.
   * @return store state information
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error reading store properties.
   */
  AnyMap getStoreInfo(String storeName, boolean includeObjectInfos) throws ObjectStoreException;

  // object handling methods

  /**
   * get the complete content of an object. Use only if you are sure that the object is small enough to fit in memory.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws NoSuchObjectException
   *           if the object does not exist
   * @throws ObjectStoreException
   *           other error reading the object
   */
  byte[] getObject(String storeName, String objectId) throws ObjectStoreException;

  /**
   * get a stream for reading the object content.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws NoSuchObjectException
   *           if the object does not exist
   * @throws ObjectStoreException
   *           other error reading the object
   */
  InputStream readObject(String storeName, String objectId) throws ObjectStoreException;

  /**
   * write the data to the given store and object. If the object exists already, it is overwritten.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           other error reading the object
   */
  void putObject(String storeName, String objectId, byte[] data) throws ObjectStoreException;

  /**
   * append the data to the given store and object. If the object exists already, the new data is appended at the end of
   * the object. If the object does not exist already, it is created.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           other error reading the object
   */
  void appendToObject(String storeName, String objectId, byte[] data) throws ObjectStoreException;

  /**
   * Prevent further append calls to this object. It does not prevent the object from being removed. It is irrelevant
   * for objects that were not created using {@link #appendToObject(String, String, byte[])}. The operation is optional:
   * An implementation may choose not to support it, in this case it should just ignore the call and not throw an
   * exception. However, in some pathological situations this may lead to data loss, because data could be appended for
   * processing when the object is already considered as finished and the next processing steps have been started and
   * miss the latest appends.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   */
  void finishObject(String storeName, String objectId) throws ObjectStoreException;

  /**
   * open a stream for creating a data object and writing it. If the object exists already, it is overwritten. The
   * object will not become visible in the store until the stream is closed. To prevent the object from becoming visible
   * then (because some error has occurred and the object is invalid), call {@link StoreOutputStream#abort()} before
   * closing the stream.
   * 
   * @return stream for writing data to the object.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error creating the object.
   */
  StoreOutputStream writeObject(String storeName, String objectId) throws ObjectStoreException;

  /**
   * @return true if object exists in store, else false.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error checking object existence.
   */
  boolean existsObject(String storeName, String objectId) throws ObjectStoreException;

  /**
   * remove the object from the store. No exception is thrown if the object does not exist in the store currently
   * (however, there is one if the store itself does not exist).
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error removing the object.
   */
  void removeObject(String storeName, String objectId) throws ObjectStoreException;

  /**
   * remove all objects from the store with an ID starting with the given prefix.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist
   * @throws ObjectStoreException
   *           error removing the object.
   */
  void removeObjects(String storeName, String objectIdPrefix) throws ObjectStoreException;

  /**
   * get information about a single object.
   * 
   * @return information about the object.
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist.
   * @throws NoSuchObjectException
   *           if the object does not exist in the store.
   * @throws ObjectStoreException
   *           error reading the object info.
   */
  StoreObject getObjectInfo(String storeName, String objectId) throws ObjectStoreException;

  /**
   * remove all stores from the service. All data will be lost, no store will exist anymore.
   * 
   * @throws ObjectStoreException
   *           error removing the data.
   */
  void removeAllStores() throws ObjectStoreException;

  /**
   * Remove all objects from the named store. The store itself will continue to exist with the same properties as
   * before, it will just be empty as if newly created.
   * 
   * @throws InvalidStoreNameException
   *           if the store name is not valid.
   * @throws NoSuchStoreException
   *           if the store does not exist.
   * @throws ObjectStoreException
   *           error clearing the store.
   */
  void clearStore(String storeName) throws ObjectStoreException;

}
