/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.input;

import java.lang.reflect.Constructor;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.objectstore.ObjectStoreService;

import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.io.IODataObject;
import org.eclipse.smila.taskworker.io.IODataObjects;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * Returned by {@link org.eclipse.smila.taskworker.TaskContext#getInputs()} this class provides access to the data
 * objects associated by a task to the input slots of a worker. It can create different kinds of {@link Input} wrappers
 * on these data objects that make access for the workers easier. You can create only a single input wrapper for each
 * data object. On the second call, only null will be returned.
 * 
 * 
 * @see StreamInput
 * @see RecordInput
 * @see KvoInput
 */
public class Inputs extends IODataObjects {
  /** create instance. */
  public Inputs(final Map<String, List<BulkInfo>> dataObjects, final ObjectStoreService objectStore) {
    super(dataObjects, objectStore);
  }

  /**
   * Generic method to create input wrapper for the first data object in a slot. See
   * {@link #getAsInput(String, int, Class)} for details, this method just calls this method with <code>index = 0</code>
   * 
   * @return first data object in given slot prepared for custom access. Null, if slot is not set, or an input wrapper
   *         has already been created earlier.
   * @throws IllegalArgumentException
   *           if the inputClass cannot be instantiated.
   */
  public <T extends Input> T getAsInput(final String slotName, final Class<T> inputClass) {
    return getAsInput(slotName, 0, inputClass);
  }

  /**
   * Generic method to create input wrappers. Any predefined subclass of {@link Input} can be used as well as custom
   * defined class if you need it (be sure to implement the necessary constructor). For example, to get an
   * Key-Value-Object wrapper for the first object in slot "kvoInputSlot", use
   * <code>KVOInput kvo = taskContext.getInputs().getAsInput("kvoInputSlot", KVOInput.class);</code>
   * 
   * @see StreamInput
   * @see RecordInput
   * @see KvoInput
   * 
   * @return n-th data object in given slot prepared for custom access. Null, if slot is not set, has less objects, or a
   *         wrapper has already been created earlier.
   * @throws IllegalArgumentException
   *           if the inputClass cannot be instantiated.
   */
  public <T extends Input> T getAsInput(final String slotName, final int index, final Class<T> inputClass) {
    if (canCreate(slotName, index)) {
      final BulkInfo dataObject = getDataObject(slotName, index);
      try {
        final Constructor<T> inputConstructor = inputClass.getConstructor(BulkInfo.class, ObjectStoreService.class);
        final T input = inputConstructor.newInstance(dataObject, getObjectStore());
        putIOData(slotName, index, input);
        return input;
      } catch (final Exception ex) {
        throw new IllegalArgumentException("Canot create instance of class " + inputClass, ex);
      }
    }
    return null;
  }

  /**
   * @return first data object in given slot prepared for stream access. Null, if slot is not set, or a wrapper has
   *         already been created earlier.
   */
  public StreamInput getAsStreamInput(final String slotName) {
    return getAsStreamInput(slotName, 0);
  }

  /**
   * @return n-th data object in given slot prepared for stream access. Null, if slot is not set, has less objects, or a
   *         wrapper has already been created earlier.
   */
  public StreamInput getAsStreamInput(final String slotName, final int index) {
    return getAsInput(slotName, index, StreamInput.class);
  }

  /**
   * @return first data object in given slot prepared for record access. Null, if slot is not set,or a wrapper has
   *         already been created earlier.
   */
  public RecordInput getAsRecordInput(final String slotName) {
    return getAsRecordInput(slotName, 0);
  }

  /**
   * @return n-th data object in given slot prepared for record access. Null, if slot is not set, has less objects, or a
   *         wrapper has already been created earlier.
   */
  public RecordInput getAsRecordInput(final String slotName, final int index) {
    return getAsInput(slotName, index, RecordInput.class);
  }

  /**
   * close all input wrapper objects. Called by the WorkerManager after the task has been finished.
   */
  public void closeAll() {
    for (final List<IODataObject> inputs : getIOData().values()) {
      for (final IODataObject input : inputs) {
        if (input != null) {
          ((Input) input).close();
        }
      }
    }
  }

  /** aggregate counters of input slots. Called by the WorkerManager after the task has been finished. */
  public void addInputCounters(final Map<String, Number> counters) {
    addCounters(counters, Counters.INPUT);
  }

}
