/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.admin;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.common.params.CoreAdminParams;
import org.apache.solr.common.params.CoreAdminParams.CoreAdminAction;
import org.eclipse.smila.solr.server.SolrServers;
import org.eclipse.smila.solr.util.SolrHelper;

/**
 * The SolrBaseAdmin class.
 * 
 * @author pwissel
 * 
 */
public abstract class SolrBaseAdmin implements SolrAdmin {

  /** The log. */
  protected final Log _log = LogFactory.getLog(SolrAdmin.class);

  /** solr helper. */
  protected final SolrHelper _helper;

  /** solr servers manager. */
  private final SolrServers _servers;

  /** cache for field names defined in cores. */
  private final Map<String, List<String>> _cachedFieldNames = new HashMap<String, List<String>>();

  /**
   * Constructor.
   * 
   * @param helper
   *          the SolrHelper.
   */
  public SolrBaseAdmin(SolrHelper helper, SolrServers servers) {
    _helper = helper;
    _servers = servers;
  }

  /** get manager of {@link SolrServer}s. */
  public SolrServers getServers() {
    return _servers;
  }

  /**
   * Build the Solr parameter map.
   * 
   * @param action
   *          the action to perform.
   * @param core
   *          the core name or null.
   * @param other
   *          the other core name or null.
   * @return
   */
  protected Map<String, String[]> buildAdminParams(final CoreAdminAction action, final String core,
    final String other) {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    final String[] actionParam = { action.toString() };
    params.put(CoreAdminParams.ACTION, actionParam);
    if (StringUtils.isNotBlank(core)) {
      final String[] coreParam = { core };
      params.put(CoreAdminParams.CORE, coreParam);
    }
    if (StringUtils.isNotBlank(other)) {
      final String[] otherCoreParam = { other };
      params.put(CoreAdminParams.OTHER, otherCoreParam);
    }
    return params;
  }

  @Override
  public List<String> getFieldNames(String coreName) throws Exception {
    List<String> fieldNames = _cachedFieldNames.get(coreName);
    if (fieldNames == null) {
      synchronized (_cachedFieldNames) {
        fieldNames = readFieldNames(coreName);
        _cachedFieldNames.put(coreName, fieldNames);
      }
    }
    return fieldNames;
  }

  /** actually read field names for given core. */
  protected abstract List<String> readFieldNames(String coreName) throws Exception;

  @Override
  public void clearCoreCaches() {
    synchronized (_cachedFieldNames) {
      _cachedFieldNames.clear();
      _servers.removeAllServers();
    }
  }

  @Override
  public void clearCoreCache(String coreName) {
    synchronized (_cachedFieldNames) {
      _cachedFieldNames.remove(coreName);
      _servers.removeSolrServer(coreName);
    }
  }

}
