/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.util;

import org.apache.commons.lang.NotImplementedException;
import org.apache.solr.common.params.FacetParams;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.solr.SolrConstants;

/**
 * The FacetQueryConfigAdapter is a helper class to configure a facetby object in SMILA record. While creating an
 * instance of this class you have to select a facet type: Facet.SOLR is mandatory once to enable faceting. You can
 * additionally set global parameter to this object. For every facet you want to get use the other Facet enum types. You
 * are able to set parameter to this objects too (on a per field basis). The Facet.QUERY type also needs String array
 * with the query. What types and parameter mean is described here: http://wiki.apache.org/solr/SimpleFacetParameters.
 * 
 * @author pwissel
 * @deprecated as of 1.0
 */
@Deprecated
public class FacetQueryConfigAdapter {

  /**
   * Facet type enum.
   * 
   * @author pwissel
   * 
   */
  public enum FacetType {
    /**
     * facet.date
     */
    DATE,
    /**
     * facet.field
     */
    FIELD,
    /**
     * global (for all other facets or default).
     */
    GLOBAL,
    /**
     * facet.query
     */
    QUERY;
  }

  /**
   * Facet sort enum.
   * 
   * @author pwissel
   * 
   */
  public enum FacetSort {
    /**
     * Count.
     */
    COUNT,
    /**
     * Index.
     */
    INDEX;
    @Override
    public String toString() {
      return this.name().toLowerCase();
    };
  }

  /**
   * Facet method enum.
   * 
   * @author pwissel
   * 
   */
  public enum FacetMethod {
    /**
     * Enum.
     */
    ENUM,
    /**
     * FC.
     */
    FC;
    @Override
    public String toString() {
      return this.name().toLowerCase();
    };
  }

  /**
   * Facet date other enum.
   * 
   * @author pwissel
   * 
   */
  public enum FacetDateOther {
    /**
     * Before.
     */
    BEFORE,
    /**
     * After.
     */
    AFTER,
    /**
     * Between.
     */
    BETWEEN,
    /**
     * None.
     */
    NONE,
    /**
     * All.
     */
    ALL;
    @Override
    public String toString() {
      return this.name().toLowerCase();
    };
  }

  /**
   * The internal map.
   */
  private final AnyMap _map = DataFactory.DEFAULT.createAnyMap();

  /**
   * Constructor,
   * 
   * @param facet
   *          the facet type.
   * @param query
   *          only used for type = {@link FacetType#QUERY} is then the equivalent of a facet.query
   */
  public FacetQueryConfigAdapter(FacetType facet, String... query) {
    switch (facet) {
      case GLOBAL:
        _map.put(FacetParams.FACET, true);
        break;
      case DATE:
        _map.put(SolrConstants.FACET_TYPE, FacetParams.FACET_DATE);
        break;
      case FIELD:
        _map.put(SolrConstants.FACET_TYPE, FacetParams.FACET_FIELD);
        break;
      case QUERY:
        if (query == null || query.length == 0) {
          throw new IllegalArgumentException("The parameter query must not be null: " + facet.toString());
        }
        _map.put(SolrConstants.FACET_TYPE, FacetParams.FACET_QUERY);
        for (final String fc : query) {
          _map.getSeq(SolrConstants.FACET_QUERY_ATTR, true).add(fc);
        }
        break;
      default:
        throw new NotImplementedException("The facet is not supported: " + facet.toString());
    }
  }

  /**
   * Get the underlying any map.
   * 
   * @return the map.
   */
  public AnyMap getAnyMap() {
    return _map;
  }

  /**
   * Set facet.prefix parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetPrefix(String value) {
    _map.put(FacetParams.FACET_PREFIX, value);
    return this;
  }

  /**
   * Set facet.sort parameter.
   * 
   * @param sort
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetSort(FacetSort sort) {
    _map.put(FacetParams.FACET_SORT, sort.toString());
    return this;
  }

  /**
   * Set facet.limit parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetLimit(int value) {
    _map.put(FacetParams.FACET_LIMIT, value);
    return this;
  }

  /**
   * Set facet.offset parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetOffset(int value) {
    _map.put(FacetParams.FACET_OFFSET, value);
    return this;
  }

  /**
   * Set facet.mincount parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetMinCount(int value) {
    _map.put(FacetParams.FACET_MINCOUNT, value);
    return this;
  }

  /**
   * Set facet.missing parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetMissing(Boolean value) {
    _map.put(FacetParams.FACET_MISSING, value);
    return this;
  }

  /**
   * Set facet.method parameter.
   * 
   * @param method
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetMethod(FacetMethod method) {
    _map.put(FacetParams.FACET_METHOD, method.toString());
    return this;
  }

  /**
   * Set facet.enum.cache.minDf parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetEnumCacheMinDf(int value) {
    _map.put(FacetParams.FACET_ENUM_CACHE_MINDF, value);
    return this;
  }

  /**
   * Set facet.date.start parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetDateStart(String value) {
    _map.put(FacetParams.FACET_DATE_START, value);
    return this;
  }

  /**
   * Set facet.date.end parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetDateEnd(String value) {
    _map.put(FacetParams.FACET_DATE_END, value);
    return this;
  }

  /**
   * Set facet.date.gap parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetDateGap(String value) {
    _map.put(FacetParams.FACET_DATE_GAP, value);
    return this;
  }

  /**
   * Set facet.date.hardend parameter.
   * 
   * @param value
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetDateHardend(Boolean value) {
    _map.put(FacetParams.FACET_DATE_HARD_END, value);
    return this;
  }

  /**
   * Set facet.date.other parameter.
   * 
   * @param dateOther
   *          the value.
   * @return FacetQueryConfigAdapter.
   */
  public FacetQueryConfigAdapter setFacetDateOther(FacetDateOther dateOther) {
    _map.put(FacetParams.FACET_DATE_OTHER, dateOther.toString());
    return this;
  }

}
