/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.importing;

import java.util.Collection;

/**
 * Service interface for checking if a crawled record must be sent to the processing job.
 */
public interface DeltaService {

  /**
   * Determine delta state of record identified by sourceId and recordId. If the result is {@link State#UPTODATE} the
   * service also marks the record as visited in the current crawl job run already, so there is no need to call
   * {@link #markAsUpdated(String, String, String, String)} afterwards. In the other cases the crawler should call
   * {@link #markAsUpdated(String, String, String, String)} only if the record is actually submitted to a processing
   * job.
   * 
   * @param sourceId
   *          the name of the data source that contains the record.
   * @param recordId
   *          the record id
   * @param jobRunId
   *          the current job run id in which the crawler is running.
   * @param hashCode
   *          a string that reflects changes in the record content. This can be as simple as a version identifier if
   *          such is available in record metadata, or even a hash calculated on the actual content of the record.
   * @return an appropriate {@link State} value.
   * 
   */
  State checkState(String sourceId, String recordId, String jobRunId, String hashCode) throws DeltaException;

  /**
   * Mark the record as visited in the current crawl job run.
   * 
   * @param sourceId
   *          the name of the data source that contains the record.
   * @param recordId
   *          the record id
   * @param jobRunId
   *          the current job run id in which the crawler is running.
   * @param hashCode
   *          a string that reflects changes in the record content. This can be as simple as a version identifier if
   *          such is available in record metadata, or even a hash calculated on the actual content of the record.
   * 
   */
  void markAsUpdated(String sourceId, String recordId, String jobRunId, String hashCode) throws DeltaException;

  /**
   * delete all state information in the service about the given data source.
   * 
   * @param sourceId
   *          data source name.
   */
  void clearSource(String sourceId) throws DeltaException;

  /**
   * delete all state information in the service about all data sources.
   */
  void clearAll() throws DeltaException;

  /**
   * get Ids of all sources that currently have entries in the DeltaService.
   */
  Collection<String> getSourceIds() throws DeltaException;

  /**
   * @return number of entries for given source id.
   * @param countExact
   *          set to true to get an exact reault, but this may take some time. Else the service may return only an
   *          estimated value.
   */
  long countEntries(final String sourceId, boolean countExact) throws DeltaException;
}
