/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.importing.httphandler;

import java.net.MalformedURLException;
import java.util.List;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.DeltaException;

/**
 * ReST API handler for admin operations on single sources in DeltaService: get source information and clear source.
 */
public class DeltaSourceHandler extends ADeltaHandler {

  /** uri parameter name to force exact counting of entries. */
  private static final String FLAG_COUNTEXACT = "countExact";

  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "DELETE".equals(method);
  }

  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final String sourceId = getSourceId(requestUri);
    if ("DELETE".equals(method)) {
      return clearSource(sourceId);
    } else {
      return getSourceInfo(requestUri, sourceId, inputRecord);
    }
  }

  /** get source ID from request URI. */
  private String getSourceId(final String requestUri) {
    final List<String> dynamics = getDynamicUriParts(requestUri);
    if (dynamics.isEmpty()) {
      throw new IllegalArgumentException("Invalid URI " + requestUri + " for this handler, no source ID found.");
    }
    return dynamics.get(0);
  }

  /** remove all sources. */
  private Object clearSource(final String sourceId) throws DeltaException {
    _deltaService.clearSource(sourceId);
    return null;
  }

  /** get source info. */
  private AnyMap getSourceInfo(final String requestUri, final String sourceId, final Record inputRecord)
    throws DeltaException, MalformedURLException {
    final AnyMap source = DataFactory.DEFAULT.createAnyMap();
    boolean countExact = false;
    if (inputRecord != null) {
      final Any countExactParam = inputRecord.getMetadata().get(FLAG_COUNTEXACT);
      if (countExactParam != null && countExactParam.isBoolean()) {
        countExact = countExactParam.asValue().asBoolean();
      }
    }
    source.put("id", sourceId);
    source.put("count", _deltaService.countEntries(sourceId, countExact));
    return source;
  }
}
