/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.httphandler;

import java.io.ByteArrayOutputStream;
import java.io.CharConversionException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.objectstore.BadRequestException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.StoreObject;

/**
 * Read BON object from a object store object and convert it to JSON. For bulk objects it returns only the first object.
 * 
 * @author jschumacher
 * 
 */
public class StoreObjectHandler extends AStoreHandler {
  /** reader for Any objects as result. */
  private final IpcAnyReader _ipcAnyReader = new IpcAnyReader();

  /** Record writer for BON objects to store. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter(false);

  /**
   * Checks if the handler allows the HTTP method.
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @return true if the method can be used for the URI, else false.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "PUT".equals(method) || "DELETE".equals(method);
  }

  /**
   * {@inheritDoc}
   * 
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final String storeName = getStoreName(requestUri);
    final String objectId = getObjectId(requestUri);
    if ("PUT".equals(method)) {
      putBonObject(storeName, objectId, inputRecord);
      return null;
    } else if ("DELETE".equals(method)) {
      removeObject(storeName, objectId, inputRecord);
      return null;
    } else {
      if (objectId.endsWith("*")) {
        return listObjects(storeName, objectId);
      }
      try {
        return readBonBulk(storeName, objectId);
      } catch (final IllegalStateException ex) {
        return readJsonObject(storeName, objectId);
      }
    }
  }

  /**
   * list all objects with the given prefix.
   */
  protected Any listObjects(final String storeName, final String objectIdPrefix) throws ObjectStoreException {
    final AnyMap result = FACTORY.createAnyMap();
    final AnySeq objectList = FACTORY.createAnySeq();
    result.put("objects", objectList);

    final String listObjectsPrefix = objectIdPrefix.substring(0, objectIdPrefix.length() - 1);
    final Collection<StoreObject> infos = getObjectStoreService().getStoreObjectInfos(storeName, listObjectsPrefix);

    if (infos != null) {
      for (final StoreObject objectInfo : infos) {
        final AnyMap info = objectInfo.toAny();
        objectList.add(info);
      }
    }
    return result;
  }

  /**
   * try to read object as a BON bulk.
   */
  protected BinaryObjectStreamIterator readBonBulk(final String storeName, final String objectId)
    throws ObjectStoreException, IOException {
    InputStream objectStream = null;
    try {
      objectStream = getObjectStoreService().readObject(storeName, objectId);
      final BinaryObjectStreamIterator objects = new BinaryObjectStreamIterator(objectStream);
      // try to read first object: after this we'll know if the stream is empty or not and really contains BON.
      objects.hasNext();
      return objects;
    } catch (final IllegalStateException ex) {
      // this means: that wasn't BON.
      IOUtils.closeQuietly(objectStream);
      throw ex;
    }
  }

  /**
   * try to read a single JSON object from the object store object.
   */
  protected Any readJsonObject(final String storeName, final String objectId) throws ObjectStoreException,
    IOException {
    final InputStream objectStream = getObjectStoreService().readObject(storeName, objectId);
    try {
      return _ipcAnyReader.readJsonStream(objectStream);
    } catch (final CharConversionException ex) {
      throw new BadRequestException("Object '" + objectId + "' seems to be neither in BON nor in JSON format: "
        + ex.getMessage(), ex);
    } catch (final RuntimeException ex) {
      throw new BadRequestException("Object '" + objectId + "' seems to be neither in BON nor in JSON format: "
        + ex.getMessage(), ex);
    } finally {
      IOUtils.closeQuietly(objectStream);
    }
  }

  /**
   * try to write one BON object to the object store.
   */
  protected void putBonObject(final String storeName, final String objectId, final Record inputRecord)
    throws ObjectStoreException {
    try {
      final ByteArrayOutputStream buffer = new ByteArrayOutputStream();
      _ipcRecordWriter.writeBinaryStream(inputRecord, buffer);
      getObjectStoreService().putObject(storeName, objectId, buffer.toByteArray());
    } catch (final Exception ex) {
      throw new IllegalArgumentException("Error converting the input record to BON", ex);
    }
  }

  /** Remove object, input record is not needed here but in subclass. */
  protected void removeObject(final String storeName, final String objectId, final Record inputRecord)
    throws ObjectStoreException {
    getObjectStoreService().removeObject(storeName, objectId);
  }

  /** {@inheritDoc} */
  @Override
  protected int getSuccessStatus(final String requestMethod, final String requestUri, final Record inputRecord,
    final Object resultObject) {
    if ("GET".equals(requestMethod) && resultObject instanceof BinaryObjectStreamIterator) {
      try {
        return ((BinaryObjectStreamIterator) resultObject).hasNext() ? HttpStatus.OK : HttpStatus.NO_CONTENT;
      } catch (final IOException ex) {
        // cannnot happen: hasNext() was already called in #readBonBulk(), so it should not throw here.
        throw new RuntimeException("Error reading from ObjectStore", ex);
      }
    }
    return super.getSuccessStatus(requestMethod, requestUri, inputRecord, resultObject);
  }

  /** {@inheritDoc} */
  @Override
  protected void writeResultObject(final OutputStream responseStream, final Object resultObject) throws IOException {
    if (resultObject instanceof BinaryObjectStreamIterator) {
      final BinaryObjectStreamIterator objects = (BinaryObjectStreamIterator) resultObject;
      try {
        while (objects.hasNext()) {
          _ipcRecordWriter.writeJsonStream(objects.next(), responseStream);
          responseStream.write('\n');
        }
      } catch (final IOException ex) {
        throw new RuntimeException("Error reading from ObjectStore", ex);
      } finally {
        objects.close();
      }
    } else {
      super.writeResultObject(responseStream, resultObject);
    }
  }
}
