/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.state.objectstore;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.importing.StateException;
import org.eclipse.smila.importing.StateNotFoundException;
import org.eclipse.smila.objectstore.NoSuchObjectException;
import org.eclipse.smila.objectstore.NoSuchStoreException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.utils.digest.DigestHelper;

/**
 * ObjectStore based implementation of a state service used in the jobmanager based importing framework.
 * 
 * @author scum36
 * 
 */
public class ObjectStoreStateService {

  /** key for the entry. */
  protected static final String KEY = "key";

  /** key for the objectId entry in the data object. */
  protected static final String KEY_OBJECTID = "objectId";

  /** key for the sourceId entry in the data object. */
  protected static final String KEY_SOURCEID = "sourceId";

  /** key for the descriptor entry in the data object. */
  protected static final String KEY_DESCRIPTOR = "descriptor";

  /** key for the job run ID entry in the data object. */
  protected static final String KEY_JOBRUNID = "jobRunId";

  /** root directory in ObjectStore for delta entries. */
  protected static final String ROOT_ENTRIES = "entries/";

  /** limit for number of entries to count exact. */
  private static final int LIMIT_COUNTENTRIES_EXACT = 10000;

  /** my logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** BON parser for entry values. */
  private final IpcAnyReader _parser = new IpcAnyReader();

  /** BON writer for entry values. */
  private final IpcAnyWriter _writer = new IpcAnyWriter();

  /** store configuration. */
  private final StateStoreConfiguration _configuration;

  /** reference to objectstore service. */
  private final ObjectStoreService _objectStore;

  /** store configuration. */
  private final String _storeName;

  public ObjectStoreStateService(final String storeName, final StateStoreConfiguration configuration,
    final ObjectStoreService objectStore) {
    _storeName = storeName;
    _configuration = configuration;
    _objectStore = objectStore;
  }

  public AnyMap getEntry(final String sourceId, final String objectId) throws StateException {
    final String key = createDeltaKey(sourceId, objectId);
    return readEntry(key);
  }

  public void mark(final String sourceId, final String objectId, final String jobRunId, final String descriptor)
    throws StateException {
    final String key = createDeltaKey(sourceId, objectId);
    final AnyMap entry = createDeltaEntry(sourceId, objectId, jobRunId, descriptor);
    entry.put(KEY, key);
    writeEntry(entry);
  }

  public void clearSource(final String sourceId) throws StateException {
    try {
      _objectStore.removeObjects(_storeName, createDeltaEntryBase(sourceId));
    } catch (final NoSuchStoreException ex) {
      ; // ok. there is nothing to clear
    } catch (final ObjectStoreException ex) {
      throw new StateException("Error removing all entries for source " + sourceId, ex);
    }
  }

  public void clearAll() throws StateException {
    try {
      _objectStore.clearStore(_storeName);
    } catch (final NoSuchStoreException ex) {
      ; // ok. there is nothing to clear
    } catch (final ObjectStoreException ex) {
      throw new StateException("Error clearing all entries of all sources", ex);
    }
  }

  /** @return all source ids. Throws StateNotFoundException if store doesn't exist yet. */
  public Collection<String> getSourceIds() throws StateException {
    try {
      final Collection<String> sourceIds = new ArrayList<String>();
      try {
        final Collection<String> sourceEntries = _objectStore.getPrefixes(_storeName, ROOT_ENTRIES);
        for (final String sourceEntry : sourceEntries) {
          String sourceId = sourceEntry.substring(ROOT_ENTRIES.length());
          if (sourceId.endsWith("/")) {
            sourceId = sourceId.substring(0, sourceId.length() - 1);
          }
          sourceIds.add(sourceId);
        }
      } catch (final NoSuchStoreException ex) {
        ; // ignoreo
      }
      return sourceIds;
    } catch (final ObjectStoreException ex) {
      throw new StateException("Error determining existing sources.", ex);
    }
  }

  /** count the entries of the given source. Throws StateNotFoundException if store or source do not exist yet. */
  public long countEntries(final String sourceId, final boolean countExact) throws StateException {
    if (!hasSource(sourceId)) {
      throw new StateNotFoundException("Source '" + sourceId + "' doesn't exist in store '" + _storeName + "'");
    }
    try {
      long entryCount = 0;
      int shardCount = 0;
      final String sourceBase = createDeltaEntryBase(sourceId);
      final Collection<String> shardIds = _objectStore.getPrefixes(_storeName, sourceBase);
      if (shardIds != null) {
        for (final String shardId : shardIds) {
          String shardPrefix = shardId;
          if (!shardId.endsWith("/")) {
            shardPrefix += "/";
          }
          entryCount += _objectStore.countStoreObjects(_storeName, shardPrefix);
          shardCount++;
          if (!countExact && entryCount > LIMIT_COUNTENTRIES_EXACT) { // estimate.
            return (entryCount * shardIds.size()) / shardCount;
          }
        }
      }
      return entryCount;
    } catch (final ObjectStoreException ex) {
      throw new StateException("Error determining existing sources.", ex);
    }
  }

  /** @return whether store contains entries for given source. */
  private boolean hasSource(final String sourceId) throws StateException {
    return getSourceIds().contains(sourceId);
  }

  /** read the current entry from the object store. */
  private AnyMap readEntry(final String key) throws StateException {
    final byte[] data;
    try {
      data = readEntryIfExists(key);
    } catch (final ObjectStoreException ex) {
      throw new StateException("Error reading object " + key + " from objectstore", ex);
    }
    if (data == null) {
      return null;
    }
    try {
      return (AnyMap) _parser.readBinaryObject(data);
    } catch (final IOException ex) {
      throw new StateException("Error parsing object " + key, ex);
    }
  }

  /** try to read data from object store. If {@link NoSuchObjectException} is thrown, return null. */
  private byte[] readEntryIfExists(final String key) throws ObjectStoreException {
    try {
      return ObjectStoreRetryUtil.retryGetObject(_objectStore, _storeName, key);
    } catch (final NoSuchObjectException ex) {
      return null;
    } catch (final NoSuchStoreException ex) {
      return null;
    }
  }

  /** update the entry in the objectstore. */
  protected void writeEntry(final AnyMap value) throws StateException {
    final String key = value.getStringValue(KEY);
    byte[] data = null;
    try {
      data = _writer.writeBinaryObject(value);
    } catch (final IOException ex) {
      throw new StateException("Error converting object " + value + " for " + key + " to BON", ex);
    }
    writeEntryEnsureStore(key, data);
  }

  /** write entry, create store if it doesn't exist already. */
  private void writeEntryEnsureStore(final String key, final byte[] data) throws StateException {
    try {
      try {
        ObjectStoreRetryUtil.retryPutObject(_objectStore, _storeName, key, data);
      } catch (final NoSuchStoreException ex) {
        _log.info("Creating store '" + _storeName + "'");
        ObjectStoreRetryUtil.retryEnsureStore(_objectStore, _storeName);
        ObjectStoreRetryUtil.retryPutObject(_objectStore, _storeName, key, data);
      }
    } catch (final ObjectStoreException ex) {
      throw new StateException("Error writing object for " + key + " to objectstore", ex);
    }
  }

  /** create the object key for the delta entry in the objectstore. */
  private String createDeltaKey(final String sourceId, final String objectId) {
    final String idDigest = DigestHelper.calculateDigest(objectId);
    return createDeltaEntryBase(sourceId) + _configuration.getEntryKey(idDigest);
  }

  /** create base path for delta entries of the given source. */
  private String createDeltaEntryBase(final String sourceId) {
    return ROOT_ENTRIES + sourceId + '/';
  }

  /** create the data object to store in object store. */
  private AnyMap createDeltaEntry(final String sourceId, final String objectId, final String jobRunId,
    final String descriptor) {
    final AnyMap entry = DataFactory.DEFAULT.createAnyMap();
    entry.put(KEY_OBJECTID, objectId);
    entry.put(KEY_SOURCEID, sourceId);
    entry.put(KEY_DESCRIPTOR, descriptor);
    entry.put(KEY_JOBRUNID, jobRunId);
    return entry;
  }

}
