/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.common.definitions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;

/**
 * ValueXpression encapsulates a string with potential variable references.
 */
public final class ValueExpression {

  /** variable pattern string. */
  public static final String VARIABLE_REGEX = "\\$\\{([^\\$\\{\\s]+?)\\}";

  /** variables recognition start string. */
  private static final String VARIABLE_START = "${";

  /** the end string of a parameter. */
  private static final String VARIABLE_END = "}";

  /** variable pattern, must consist of VARIABLE_START and VARABLE_END prepared for RegEx. */
  private static final Pattern VARIABLE_PATTERN = Pattern.compile(VARIABLE_REGEX);

  /** the expression with potentially some variable references. */
  private final String _expression;

  /** the referenced variables. */
  private final List<String> _variables = new ArrayList<String>();

  /**
   * Constructs a new VariableExpression.
   * 
   * @param expression
   *          The expression string.
   */
  public ValueExpression(final String expression) {
    super();
    _expression = expression;
    findVariables();
  }

  /**
   * Evaluates an expression with a given context.
   * 
   * @param context
   *          A map with known variables, will be used to resolve variables within the expression.
   * @return An expression with each variable references exchanged by the corresponding values in the context.
   */
  public ValueExpression evaluate(final Map<String, String> context) {
    String intermediate = _expression;
    // do this more than once, maybe there is another reference as value on the context
    // e.g.
    // { "param1" : "${param2}", "param2": "real value" }
    boolean intermediateChanged = true;
    while (intermediateChanged && referencesVariables(intermediate)) {
      final String oldIntermediate = intermediate;
      for (final Entry<String, String> entry : context.entrySet()) {
        intermediate = solveVariableReference(intermediate, entry.getKey(), entry.getValue());
      }
      intermediateChanged = !oldIntermediate.equals(intermediate);
    }
    return new ValueExpression(intermediate);
  }

  /**
   * Evaluates an expression with a given context.
   * 
   * @param context
   *          A map with known variables, will be used to resolve variables within the expression. Only the string
   *          representation of toplevel values will be used to resolve the variable.
   * @return An expression with each variable references exchanged by the corresponding values in the context.
   */
  public ValueExpression evaluate(final AnyMap context) {
    String intermediate = _expression;
    // do this more than once, maybe there is another reference as value on the context
    // e.g.
    // { "param1" : "${param2}", "param2": "real value" }
    boolean intermediateChanged = true;
    while (intermediateChanged && referencesVariables(intermediate)) {
      final String oldIntermediate = intermediate;
      for (final Entry<String, Any> entry : context.entrySet()) {
        if (entry.getValue().isValue()) {
          intermediate =
            solveVariableReference(intermediate, entry.getKey(), entry.getValue().asValue().asString());
        }
      }
      intermediateChanged = !oldIntermediate.equals(intermediate);
    }
    return new ValueExpression(intermediate);
  }

  /**
   * Evaluates an expression with a given context.
   * 
   * @param context
   *          A map with known variables, will be used to resolve variables within the expression.
   * @return An expression with each variable references exchanged by the corresponding values in the context.
   */
  public ValueExpression evaluateExpr(final Map<String, ValueExpression> context) {
    String intermediate = _expression;
    // do this more than once, maybe there is another reference as value on the context
    // e.g.
    // { "param1" : "${param2}", "param2": "real value" }
    boolean intermediateChanged = true;
    while (intermediateChanged && referencesVariables(intermediate)) {
      final String oldIntermediate = intermediate;
      for (final Entry<String, ValueExpression> entry : context.entrySet()) {
        intermediate = solveVariableReference(intermediate, entry.getKey(), entry.getValue().getExpression());
      }
      intermediateChanged = !oldIntermediate.equals(intermediate);
    }
    return new ValueExpression(intermediate);
  }

  /**
   * resolves one variable within a string with its value.
   * 
   * @param expression
   *          A String containing a variable reference.
   * @param name
   *          The variable name.
   * @param value
   *          The value of the variable.
   * @return The expression with substituted value for the variabel reference.
   */
  private String solveVariableReference(final String expression, final String name, final String value) {
    StringBuilder stringBuilder = new StringBuilder(expression);
    if (name != null && !name.equals("")) {
      final String variableExpression = VARIABLE_START + name + VARIABLE_END;
      if (!value.contains(variableExpression)) {
        // to prevent infinite loops...
        int beginIndex = stringBuilder.indexOf(variableExpression);
        while (beginIndex >= 0) {
          stringBuilder = stringBuilder.replace(beginIndex, beginIndex + variableExpression.length(), value);
          beginIndex = stringBuilder.indexOf(variableExpression, beginIndex);
        }
      }
    }
    return stringBuilder.toString();
  }

  /**
   * try to extract variable values from a string that should be created from the expression. Throws
   * {@link IllegalArgumentException} if no match could be found.
   * 
   * @param evaluatedExpression
   *          value that is supposed to be created from the expression.
   * @return variable values.
   */
  public Map<String, String> extractValues(final String evaluatedExpression) {
    String extractRegex = _expression.replaceAll(VARIABLE_REGEX, "\\(.*\\)");
    extractRegex = extractRegex.replace("$", "\\$");
    extractRegex = extractRegex.replace("{", "\\{");
    extractRegex = extractRegex.replace("}", "\\}");
    final Matcher m = Pattern.compile(extractRegex).matcher(evaluatedExpression);
    final Map<String, String> values = new HashMap<String, String>();
    if (m.find()) {
      int i = 1;
      for (final String variable : _variables) {
        values.put(variable, m.group(i++));
      }
    } else {
      throw new IllegalArgumentException("'" + evaluatedExpression + "' doesn't match expression '" + _expression
        + "'");
    }
    return values;
  }

  /**
   * Checks if this valueExpression contains a references to a variable.
   * 
   * @return true if this valueExpression contains a references to a variable.
   */
  public boolean referencesVariables() {
    return referencesVariables(_expression);
  }

  /**
   * Gets the recognizes variable names in the value.
   * 
   * @return The recognized variables.
   */
  public Collection<String> getVariables() {
    return new HashSet<String>(_variables);
  }

  /**
   * Returns the expression as a String.
   * 
   * @return The expression as a String.
   */
  public String getExpression() {
    return _expression;
  }

  // ----- override section -----
  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object arg0) {
    if (arg0 instanceof ValueExpression) {
      return _expression.equals(((ValueExpression) arg0).getExpression());
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return _expression.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return _expression;
  }

  // ----- private section -----
  /**
   * find variables in the template string.
   */
  private void findVariables() {
    if (_expression != null) {
      final Matcher m = VARIABLE_PATTERN.matcher(_expression);
      while (m.find()) {
        _variables.add(m.group(1));
      }
    }
  }

  /**
   * Checks if potentialExpression contains a references to a variable.
   * 
   * @param potentialExpression
   *          a string potentially containing a reference to a variable.
   * @return true if potentialExpression contains a references to a variable.
   */
  private static boolean referencesVariables(final String potentialExpression) {
    return VARIABLE_PATTERN.matcher(potentialExpression).find();
  }
}
