/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.filesystem;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.eclipse.smila.objectstore.StoreOutputStream;

/**
 * <p>
 * Simple {@link StoreOutputStream} implementation.
 * </p>
 * 
 * <p>
 * Intermediate content is stored in a temporary directory but will not be visible until {@link #close()} is called.
 * </p>
 */
public class SimpleStoreOutputStream extends StoreOutputStream {

  /** closed flag. */
  private boolean _closed;

  /** the wrapped {@link FileOutputStream}. */
  private final FileOutputStream _os;

  /** temporary file (so that before the stream is closed, the file won't be visible. */
  private final File _tempFile;

  /** the final (and visible) file. */
  private final File _finalFile;

  /** should files be locked? */
  private final boolean _fileLockingRequested;

  /**
   * constructs a SimpleStoreOutputStream.
   * 
   * @throws FileNotFoundException
   */
  public SimpleStoreOutputStream(final File tempFile, final File finalFile, final boolean fileLocking)
    throws FileNotFoundException {
    super();
    _tempFile = tempFile;
    _finalFile = finalFile;
    _os = new FileOutputStream(_tempFile);
    _fileLockingRequested = fileLocking;
  }

  /** {@inheritDoc} */
  @Override
  public void abort() {
    IOUtils.closeQuietly(_os);
    FileUtils.deleteQuietly(_tempFile);
    _closed = true;
  }

  /** {@inheritDoc} */
  @Override
  public void write(final int b) throws IOException {
    _os.write(b);
  }

  /** {@inheritDoc} */
  @Override
  public void write(final byte[] b) throws IOException {
    _os.write(b);
  }

  /** {@inheritDoc} */
  @Override
  public void write(final byte[] b, final int off, final int len) throws IOException {
    _os.write(b, off, len);
  }

  /** {@inheritDoc} */
  @Override
  public void close() throws IOException {
    try {
      if (_closed) {
        return;
      }
      _os.close();

      final AtomicInteger lock = new AtomicInteger(0);
      boolean written = false;
      do {
        AtomicInteger currentLock = SimpleObjectStoreService.LOCK_MAP.putIfAbsent(_finalFile, lock);
        if (currentLock == null) {
          currentLock = lock;
        }
        currentLock.incrementAndGet();
        synchronized (currentLock) {
          // check that no one deleted the lock...
          if (SimpleObjectStoreService.LOCK_MAP.get(_finalFile) == currentLock) {
            written = makeFileVisible();
            // clean up locks...
            if (currentLock.decrementAndGet() <= 0) {
              SimpleObjectStoreService.LOCK_MAP.remove(_finalFile);
            }
          }
        }
      } while (!written);
    } finally {
      FileUtils.deleteQuietly(_tempFile);
    }
  }

  /**
   * Makes file visible by copying it from the hidden store to the visible one and removing the shadow file.
   * 
   * @return 'true' if the operation succeeded, 'false' if not and should be retried.
   * @throws IOException
   *           operation failed
   */
  protected boolean makeFileVisible() throws IOException {
    boolean written;
    FileLock fileLock = null;
    // be sure the file to be locked really exists...
    _finalFile.getParentFile().mkdirs();
    _finalFile.createNewFile();
    final FileOutputStream os = new FileOutputStream(_finalFile);
    final FileChannel fc = os.getChannel();
    if (_fileLockingRequested) {
      fileLock = fc.lock();
    }
    try {
      final FileInputStream is = new FileInputStream(_tempFile);
      final FileChannel ifc = is.getChannel();
      int numOfBytesTransferred = 0;
      long writeByteCount = 0;
      final long fileSize = _tempFile.length();
      do {
        writeByteCount = ifc.transferTo(0, fileSize - writeByteCount, fc);
        if (writeByteCount > 0) {
          numOfBytesTransferred += writeByteCount;
        }
      } while (numOfBytesTransferred < fileSize);
      ifc.close();
      is.close();
      _closed = true;
      written = true;
    } finally {
      if (fileLock != null) {
        fileLock.release();
      }
      if (fc != null) {
        fc.close();
      }
      IOUtils.closeQuietly(os);
    }
    return written;
  }

  /** {@inheritDoc} */
  @Override
  public void flush() throws IOException {
    _os.flush();
  }
}
