/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.outputs;

import java.io.IOException;

import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.taskworker.io.IODataObject;
import org.eclipse.smila.taskworker.output.Output;

/** base class. for bulk output. */
public abstract class BulkOutput {

  /** bulk writer. */
  protected Output _bulk;

  /** the job name. */
  private final String _jobName;

  /** the ID of currently open bulk. */
  private String _currentBulkId;

  /** bulk type. */
  private final BulkType _bulkType;

  /** Point in time when the current bulk was started. */
  private final long _bulkStartTime = System.currentTimeMillis();

  /** Point in time when the current bulk was last written. */
  private long _lastModificationTime = System.currentTimeMillis();

  /** limit for bulk size. */
  private long _bulkSizeLimit = -1;

  /**
   * timeout in ms to commit bulk. this depends on bulk data and configuration (incremental insert, delete or bulk
   * indexing).
   */
  private long _commitTimeout = -1;

  /** create instance. */
  public BulkOutput(final String jobName, final String bulkId, final BulkType bulkType) {
    _jobName = jobName;
    _currentBulkId = bulkId;
    _bulkType = bulkType;
  }

  /** @return the job name. */
  public String getJobName() {
    return _jobName;
  }

  /** @return the current bulk id. */
  public String getCurrentBulkId() {
    return _currentBulkId;
  }

  /**
   * @param currentBulkId
   *          the current bulk id.
   */
  public void setCurrentBulkId(final String currentBulkId) {
    this._currentBulkId = currentBulkId;
  }

  /** @return the bulk type. */
  public BulkType getBulkType() {
    return _bulkType;
  }

  /** @return the bulk. */
  public Output getBulk() {
    return _bulk;
  }

  /**
   * @param bulk
   *          output bulk to set.
   */
  public void setBulk(final Output bulk) {
    this._bulk = bulk;
  }

  /** @return the bulk start time. */
  public long getBulkStartTime() {
    return _bulkStartTime;
  }

  /** @return the last modification time. */
  public long getLastModificationTime() {
    return _lastModificationTime;
  }

  /**
   * @param lastModificationTime
   *          the last modification time to set.
   */
  public void setLastModificationTime(final long lastModificationTime) {
    this._lastModificationTime = lastModificationTime;
  }

  /** @return the commit timeout in ms. */
  public long getCommitTimeout() {
    return _commitTimeout;
  }

  /**
   * @return <code>true</code> if the bulk has timed out (if a timeout is set).
   */
  public boolean isTimedOut() {
    return _commitTimeout >= 0 && System.currentTimeMillis() - _lastModificationTime >= _commitTimeout;
  }

  /**
   * @param commitTimeout
   *          the commit timeout (in ms) to set.
   */
  public void setCommitTimeout(final long commitTimeout) {
    this._commitTimeout = commitTimeout;
  }

  /**
   * Set the bulk size limit (in bytes).
   * 
   * @param bulkSizeLimit
   */
  public void setBulkSizeLimit(final long bulkSizeLimit) {
    _bulkSizeLimit = bulkSizeLimit;
  }

  /**
   * @return the bulk size limit (in bytes).
   */
  public long getBulkSizeLimit() {
    return _bulkSizeLimit;
  }

  /**
   * @return <code>true</code> if the bulk has grown beyond its limit (if any).
   */
  public boolean hasGrownBeyondLimit() {
    return _bulkSizeLimit >= 0 && getBytesWritten() > _bulkSizeLimit;
  }

  /**
   * committing the output.
   * 
   * @throws IOException
   * @throws ObjectStoreException
   */
  public void commit() throws BulkbuilderException {
    try {
      _bulk.commit();
    } catch (final IOException ex) {
      throw new BulkbuilderException(ex.getClass().getSimpleName() + " committing bulk " + getCurrentBulkId(), ex);
    } catch (final ObjectStoreException ex) {
      throw new BulkbuilderException(ex.getClass().getSimpleName() + " committing bulk " + getCurrentBulkId(), ex);
    }
  }

  /** access output object. */
  public IODataObject getOutput() {
    return _bulk;
  }

  /** put a record to the bulk. */
  public abstract void addRecord(final Record record) throws BulkbuilderException;

  /** get the number of bytes written. */
  public abstract long getBytesWritten();
}
