/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.impl.DefaultDataFactoryImpl;

/**
 * utility class for handling / conversion of Any objects.
 * 
 * Hint: The Any-to-JSON conversion for Date(Time)s is not symmetric. If the Any object contains Date(Time) values, they
 * will be serialized to String values in simple timestamp format. But when parsing JSON to an Any object, Date(Time)
 * strings will not be recognized anymore, but just read as String values.
 */
public final class AnyUtil {

  /** Immutable empty AnyMap instance. */
  public static final AnyMap EMPTY_MAP = DefaultDataFactoryImpl.IMMUTABLE_EMPTY_MAP;

  /**
   * prevent instance creation.
   */
  private AnyUtil() {
    // prevent instance creation
  }

  /**
   * Converts an Any object into a native java object.
   * 
   * @param any
   *          the Any object
   * @return a Pojo
   */
  public static Object anyToNative(final Any any) {
    if (any.isMap()) {
      final LinkedHashMap<String, Object> map = new LinkedHashMap<String, Object>();
      for (final Iterator<String> kIt = ((AnyMap) any).keySet().iterator(); kIt.hasNext();) {
        final String key = kIt.next();
        map.put(key, anyToNative(((AnyMap) any).get(key)));
      }
      return map;
    } else if (any.isSeq()) {
      final ArrayList<Object> list = new ArrayList<Object>();
      for (final Iterator<Any> aIt = any.iterator(); aIt.hasNext();) {
        final Any a = aIt.next();
        list.add(anyToNative(a));
      }
      return list;
    } else if (any.isString()) {
      return ((Value) any).asString();
    } else if (any.isLong()) {
      return ((Value) any).asLong();
    } else if (any.isDouble()) {
      return ((Value) any).asDouble();
    } else if (any.isBoolean()) {
      return ((Value) any).asBoolean();
    } else if (any.isDate()) {
      return ((Value) any).asDate();
    } else if (any.isDateTime()) {
      return ((Value) any).asDateTime();
    } else {
      return ((Value) any).getObject();
    }
  }

  /**
   * Converts an object to an Any.
   * 
   * @param object
   *          The object to be converted. Maps inside this object have to be Map&lt;String, ?&gt;
   * @return The converted Any
   */
  @SuppressWarnings("unchecked")
  public static Any objectToAny(final Object object) {
    Any value = null;
    if (object instanceof Any) {
      return (Any) object;
    } else if (object instanceof Map<?, ?>) {
      value = mapToAny((Map<String, Object>) object);
    } else if (object instanceof List<?>) {
      value = listToAny((List<Object>) object);
    } else {
      value = scalarObjectToAny(object);
    }
    return value;
  }

  /**
   * Converts a list to an AnySeq object.
   * 
   * @param objectList
   *          The list of objects to convert.
   * @return An AnySeq containing the objects as Any objects.
   */
  private static AnySeq listToAny(final List<Object> objectList) {
    AnySeq anySeq = null;
    if (objectList != null) {
      anySeq = DataFactory.DEFAULT.createAnySeq();
      for (final Object obj : objectList) {
        anySeq.add(objectToAny(obj));
      }
    }
    return anySeq;
  }

  /**
   * Converts a scalar object to a Value object.
   * 
   * @param obj
   *          The object to convert.
   * @return A Value representing the object.
   */
  private static Any scalarObjectToAny(final Object obj) {
    return DataFactory.DEFAULT.autoConvertValue(obj);
  }

  /**
   * Converts a map to an AnyMap object.
   * 
   * @param map
   *          The map (String to Object) to convert.
   * @return An AnyMap representing the map with all Objects converted to Any.
   */
  private static AnyMap mapToAny(final Map<String, Object> map) {
    AnyMap anyMap = null;
    if (map != null) {
      anyMap = DataFactory.DEFAULT.createAnyMap();
      for (final Entry<String, Object> entry : map.entrySet()) {
        anyMap.put(entry.getKey(), objectToAny(entry.getValue()));
      }
    }
    return anyMap;
  }

  /**
   * get value for given path(list of keys) from AnyMap object. This methods throws no exception, if the path not exists
   * an empty Any is the result.
   * 
   * @param any
   *          the Any object.
   * @param path
   *          path to the entry.
   * @return value associated to the path.
   */
  public static Any saveGet(final Any any, final String[] path) {
    if (path.length == 0) {
      return DataFactory.DEFAULT.createAnyMap();
    }
    try {
      Any current = any;
      for (final String key : path) {
        if (current.isMap()) {
          current = ((AnyMap) current).get(key);
        } else {
          current = null;
        }
      }
      if (current == null) {
        return DataFactory.DEFAULT.createStringValue("undef");
      } else {
        return current;
      }
    } catch (final Exception e) {
      return DataFactory.DEFAULT.createStringValue("undef");
    }
  }

  /**
   * convert an exception to an any object.
   * 
   * @param e
   *          exception to convert
   * @return any representation of exception
   */
  public static AnyMap exceptionToAny(final Throwable e) {
    return exceptionToAny(e, new HashSet<String>());
  }

  /**
   * convert an exception to an any object. stop in stacktrace printing when hitting known lines again.
   * 
   * @param e
   *          exception to convert
   * @param visitedLines
   *          lines that have been added to stacktraces before.
   * @return any representation of exception
   */
  private static AnyMap exceptionToAny(final Throwable e, final Collection<String> visitedLines) {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("type", e.getClass().getName());
    if (e.getMessage() != null) {
      any.put("message", e.getMessage());
    }
    final AnySeq st = DataFactory.DEFAULT.createAnySeq();
    for (final StackTraceElement stElement : e.getStackTrace()) {
      final String line = stElement.toString();
      st.add(line);
      if (!visitedLines.add(line)) {
        st.add("...");
        break;
      }
    }
    any.put("at", st);
    if (e.getCause() != null && e.getCause() != e) {
      any.put("causedBy", exceptionToAny(e.getCause(), visitedLines));
    }
    return any;
  }
}
